# frozen_string_literal: true

# Utility methods for working with YANG schemas
module PuppetX::Puppetlabs::Netconf::YangUtils
  module_function

  # Unescape a YANG description string
  # Handles escaped quotes and backslashes according to YANG specification
  #
  # @param description [String] The escaped description string
  # @return [String] The unescaped description
  def unescape_yang_description(description)
    return description unless description

    description
      .gsub(%r{\\"}, '"')      # Unescape quotes
      .gsub(%r{\\\\}, '\\')    # Unescape backslashes
      .gsub(%r{\\n}, "\n")     # Unescape newlines
      .gsub(%r{\\t}, "\t")     # Unescape tabs
      .gsub(%r{\\r}, "\r")     # Unescape carriage returns
  end

  # Extract prefix from YANG schema text
  # Handles both quoted and unquoted prefix values
  #
  # @param schema_text [String] The YANG schema text
  # @return [String, nil] The prefix value or nil if not found
  def extract_prefix(schema_text)
    return nil unless schema_text

    # Match either quoted or unquoted prefix value
    return unless schema_text =~ %r{prefix\s+(?:"([^"]+)"|([^\s;]+))}

    Regexp.last_match(1) || Regexp.last_match(2)
  end

  # Extract import statements from YANG schema text
  # Handles both quoted and unquoted module names
  #
  # @param schema_text [String] The YANG schema text
  # @return [Array<String>] List of imported module names
  def extract_imports(schema_text)
    return [] unless schema_text

    # Match either quoted or unquoted import values
    schema_text.scan(%r{import\s+(?:"([^"]+)"|([^\s;\{]+))}).map { |m| m[0] || m[1] }
  end

  # Extract namespace from YANG schema text
  #
  # @param schema_text [String] The YANG schema text
  # @return [String, nil] The namespace URI or nil if not found
  def extract_namespace(schema_text)
    return nil unless schema_text

    schema_text[%r{namespace\s+"([^"]+)"}, 1]
  end

  # Extract description from YANG schema text
  # Handles multi-line descriptions with escaped characters
  #
  # @param schema_text [String] The YANG schema text
  # @param max_length [Integer] Maximum length of description to return
  # @return [String, nil] The description or nil if not found
  def extract_description(schema_text, max_length = 500)
    return nil unless schema_text

    # Match multi-line description with escaped quotes
    return unless schema_text =~ %r{description\s+"((?:[^"\\]|\\.)*)"}m

    desc = Regexp.last_match(1)
    desc = unescape_yang_description(desc)
    desc = desc.strip[0..max_length] if max_length
    desc
  end

  # Find similar schema identifiers based on partial matching
  # Safely handles edge cases like empty strings or identifiers starting with dashes
  #
  # @param target_identifier [String] The identifier to find similarities for
  # @param schemas [Array<Hash>] List of schema hashes with 'identifier' keys
  # @param max_results [Integer] Maximum number of similar results to return
  # @return [Array<String>] List of similar schema identifiers
  def find_similar_schemas(target_identifier, schemas, max_results = 5)
    return [] unless target_identifier && !target_identifier.empty?
    return [] unless schemas&.is_a?(Array)

    # Get the first part of the identifier, handling edge cases
    parts = target_identifier.split('-').reject(&:empty?)
    return [] if parts.empty?

    search_term = parts.first.downcase

    similar = schemas.select do |schema|
      next false unless schema && schema['identifier']

      schema['identifier'].downcase.include?(search_term)
    end

    similar.map { |s| s['identifier'] }.compact.first(max_results)
  end

  # Sanitize user input for safe inclusion in error messages
  #
  # @param input [String] The user input to sanitize
  # @return [String] The sanitized input safe for error messages
  def sanitize_for_error(input)
    return '' unless input

    # Remove control characters and limit length
    input.to_s.gsub(%r{[[:cntrl:]]}, '').slice(0, 100)
  end
end
