#!/usr/bin/env ruby
# frozen_string_literal: true

# NETCONF Discovery Task
#
# This task provides discovery of NETCONF device capabilities and YANG schemas.
# Default action lists capabilities and available schemas without content.
# Use action='get_schemas' with identifiers parameter to fetch specific schema content.
#
# Usage:
#   bolt task run edgeops::netconf_discover --targets netconf://device
#   bolt task run edgeops::netconf_discover action=get_schemas identifiers='["openconfig-interfaces","openconfig-system"]' --targets netconf://device

require 'json'
require 'rexml/document'

# Include the task helper to set up load paths
require_relative '../../edgeops/files/task_helper.rb'

# Now require the netconf libraries
require 'puppet_x/puppetlabs/netconf/session'
require 'puppet_x/puppetlabs/netconf/yang_utils'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  action = session.task_params['action'] || 'list'
  identifiers = session.task_params['identifiers'] || []

  session.logger.info("Running NETCONF discovery with action: #{action}")

  if action == 'get_schemas'
    # Get specific schemas with content
    if identifiers.empty?
      raise ArgumentError, "The 'identifiers' parameter is required when action='get_schemas'. Provide an array of schema identifiers to fetch."
    end

    schemas_with_content = []
    failed_schemas = []

    identifiers.each_with_index do |identifier, index|
      session.logger.info("Fetching schema #{index + 1}/#{identifiers.length}: #{identifier}")

      begin
        # First check if schema exists
        all_schemas = session.list_schemas
        schema_info = all_schemas.find { |s| s['identifier'] == identifier }

        unless schema_info
          session.logger.warn("Schema '#{identifier}' not found on device")
          failed_schemas << { 'identifier' => identifier, 'error' => 'Schema not found' }
          next
        end

        version = schema_info['version']
        format = schema_info['format'] || 'yang'

        schema_content = session.get_schema(identifier, version, format)

        # Extract the schema text from the structured response
        schema_text = nil
        if schema_content.is_a?(Array)
          data_elem = schema_content.find { |elem| elem.name == 'data' }
          schema_text = data_elem.text if data_elem
        end

        if schema_text && !schema_text.empty?
          # Extract metadata
          metadata = {}
          if format == 'yang' && schema_text
            metadata['namespace'] = PuppetX::Puppetlabs::Netconf::YangUtils.extract_namespace(schema_text)
            metadata['prefix'] = PuppetX::Puppetlabs::Netconf::YangUtils.extract_prefix(schema_text)
            metadata['imports'] = PuppetX::Puppetlabs::Netconf::YangUtils.extract_imports(schema_text)
            metadata['description'] = PuppetX::Puppetlabs::Netconf::YangUtils.extract_description(schema_text, 500)
            metadata['has_containers'] = schema_text.include?('container ')
            metadata['has_lists'] = schema_text.include?('list ')
            metadata['has_rpcs'] = schema_text.include?('rpc ')
            metadata['has_notifications'] = schema_text.include?('notification ')
            metadata['has_groupings'] = schema_text.include?('grouping ')
            metadata['yang_version'] = schema_text[%r{yang-version\s+"?([^"\s;]+)}, 1]
          end

          schemas_with_content << {
            'identifier' => identifier,
            'version' => version,
            'format' => format,
            'metadata' => metadata,
            'content' => schema_text
          }
        else
          session.logger.warn("Failed to extract schema text for '#{identifier}'")
          failed_schemas << { 'identifier' => identifier, 'error' => 'Failed to extract content' }
        end
      rescue StandardError => e
        session.logger.warn("Failed to fetch schema '#{identifier}': #{e.message}")
        failed_schemas << { 'identifier' => identifier, 'error' => e.message }
      end
    end

    session.logger.info("Successfully fetched #{schemas_with_content.length} of #{identifiers.length} schemas")

    # Build result for get_schemas action
    result = {
      'action' => 'get_schemas',
      'requested' => identifiers.length,
      'fetched' => schemas_with_content.length,
      'schemas' => schemas_with_content
    }

    result['failed'] = failed_schemas unless failed_schemas.empty?

    session.report_result(result)

  else
    # Default action: list capabilities and schemas (without content)

    # Step 1: Get and parse capabilities
    capabilities = session.server_capabilities

    parsed_caps = {
      'base_netconf' => [],
      'yang_modules' => [],
      'other_capabilities' => []
    }

    capabilities.each do |cap|
      if cap.include?('urn:ietf:params:netconf:base')
        version = cap[%r{base:(\d+\.\d+)}, 1]
        parsed_caps['base_netconf'] << {
          'version' => version,
          'full_uri' => cap
        }
      elsif cap.include?('module=')
        module_info = {}
        module_info['identifier'] = cap[%r{module=([^&]+)}, 1]
        module_info['revision'] = cap[%r{revision=([^&]+)}, 1]
        module_info['features'] = cap.scan(%r{features=([^&]+)}).flatten
        module_info['deviations'] = cap.scan(%r{deviations=([^&]+)}).flatten
        module_info['namespace'] = cap[%r{^([^?]+)}, 1]
        parsed_caps['yang_modules'] << module_info
      else
        parsed_caps['other_capabilities'] << cap
      end
    end

    parsed_caps['yang_modules'].sort_by! { |m| m['identifier'] || '' }

    capability_summary = {
      'netconf_versions' => parsed_caps['base_netconf'].map { |c| c['version'] }.compact,
      'supports' => {
        'candidate' => session.supports_candidate?,
        'confirmed_commit' => session.supports_confirmed_commit?,
        'validate' => session.supports_validate?,
        'startup' => session.supports_startup?,
        'xpath' => capabilities.any? { |c| c.include?('xpath') },
        'writable_running' => capabilities.any? { |c| c.include?('writable-running') },
        'rollback' => capabilities.any? { |c| c.include?('rollback') }
      }
    }

    # Step 2: Get all available schemas (just metadata, no content)
    all_schemas = session.list_schemas
    all_schemas.sort_by! { |s| s['identifier'] || '' }

    session.logger.info("Found #{all_schemas.length} schemas")

    # Build result for default list action
    session.report_result({
                            'action' => 'list',
      'capabilities' => {
        'summary' => capability_summary,
        'base_netconf' => parsed_caps['base_netconf'],
        'yang_modules' => parsed_caps['yang_modules'],
        'other' => parsed_caps['other_capabilities']
      },
      'schemas' => {
        'total' => all_schemas.length,
        'schemas' => all_schemas # Just metadata, no content
      }
                          })
  end
end

# Output the result as JSON for Bolt
puts result.to_json
