#!/usr/bin/env ruby
# frozen_string_literal: true

# Retrieves a specific YANG schema from a NETCONF device
#
# This task uses Session#get_schema to retrieve YANG module definitions
# from devices that support the ietf-netconf-monitoring capability.
#
# Parameters:
#   - identifier: (required) The schema identifier/module name to retrieve
#   - version: (optional) The specific version/revision to retrieve
#   - format: (optional) The schema format (default: 'yang', also supports 'yin')
#   - preview_only: (optional) Return only first 1000 characters of schema
#
# Returns:
#   - identifier: The schema identifier that was requested
#   - version: The version/revision retrieved
#   - format: The format of the schema
#   - content: The full schema text
#   - content_length: Size of the schema in bytes
#   - content_preview: First 1000 chars if preview_only is true
#   - metadata: Parsed metadata from YANG schema (imports, namespaces, etc.)

require 'json'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Get task parameters
  identifier = session.task_params['identifier']
  version = session.task_params['version']
  format = session.task_params['format'] || 'yang'

  # Validate required parameters
  unless identifier
    raise ArgumentError, 'identifier parameter is required - specify the YANG module name to retrieve'
  end

  # Check if device supports ietf-netconf-monitoring (required for get-schema)
  monitoring_cap = session.server_capabilities.find { |c| c.include?('urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring') }

  unless monitoring_cap
    session.logger.error('Device does not support ietf-netconf-monitoring capability')
    raise 'Device does not support ietf-netconf-monitoring capability required for get-schema operation'
  end

  # Check if module exists in capabilities
  module_in_caps = session.server_capabilities.any? { |c| c.include?("module=#{identifier}") }

  unless module_in_caps
    # Try to find similar modules
    similar_caps = session.server_capabilities.select do |c|
      c.include?('module=') && c.downcase.include?(identifier.split('-').first.downcase)
    end
    similar = similar_caps.map { |c|
      c.match(%r{module=([^&]+)})&.[](1)
    }.compact

    error_msg = "Module '#{identifier}' not found in device capabilities."
    unless similar.empty?
      error_msg += " Did you mean one of these? #{similar.first(5).join(', ')}"
    end
    raise error_msg
  end

  # Extract version from capability if not provided
  if version.nil?
    matching_cap = session.server_capabilities.find { |c| c.include?("module=#{identifier}") }
    if matching_cap&.match(%r{revision=([^&]+)})
      version = Regexp.last_match(1)
      session.logger.info("Using version from capability: #{version}")
    end
  end

  # Get the schema content
  begin
    schema_content = session.get_schema(identifier, version, format)

    # Extract the schema text from the structured response
    schema_text = nil
    if schema_content.is_a?(Array)
      # Find the data element containing the schema
      data_elem = schema_content.find { |elem| elem.name == 'data' }
      schema_text = data_elem.text if data_elem
    end

    if schema_text.nil? || schema_text.empty?
      session.logger.error('Failed to extract schema text from response')
      raise 'Failed to retrieve schema content'
    end

    unless schema_text && !schema_text.strip.empty?
      session.logger.error("No schema text found in response: #{schema_content.map(&:to_s).join}")
      raise "Failed to extract schema text for '#{identifier}'"
    end

    session.logger.info("Successfully retrieved schema '#{identifier}' (#{schema_text.length} bytes)")

    # Parse metadata if it's a YANG schema
    metadata = {}
    if format == 'yang' && schema_text
      # Extract basic YANG metadata
      metadata['has_module'] = schema_text.include?('module ')
      metadata['has_imports'] = schema_text.include?('import ')
      metadata['imports'] = schema_text.scan(%r{import\s+"?([^"\s;]+)}).flatten
      metadata['prefixes'] = schema_text.scan(%r{prefix\s+"?([^"\s;]+)}).flatten
      metadata['namespace'] = schema_text[%r{namespace\s+"([^"]+)"}, 1]
      metadata['contact'] = schema_text[%r{contact\s+"([^"]+)"}, 1]
      metadata['organization'] = schema_text[%r{organization\s+"([^"]+)"}, 1]
      metadata['description'] = schema_text[%r{description\s+"([^"]{0,200})[^"]*"}, 1]

      # Count YANG constructs
      metadata['containers'] = schema_text.scan(%r{container\s+\w+}).count
      metadata['lists'] = schema_text.scan(%r{list\s+\w+}).count
      metadata['leafs'] = schema_text.scan(%r{leaf\s+\w+}).count
      metadata['leaf_lists'] = schema_text.scan(%r{leaf-list\s+\w+}).count
      metadata['groupings'] = schema_text.scan(%r{grouping\s+\w+}).count
      metadata['typedefs'] = schema_text.scan(%r{typedef\s+\w+}).count
      metadata['rpcs'] = schema_text.scan(%r{rpc\s+\w+}).count
      metadata['notifications'] = schema_text.scan(%r{notification\s+\w+}).count
    end

    # Return the results via report_result
    session.report_result({
                            'identifier' => identifier,
      'version' => version,
      'format' => format,
      'metadata' => metadata,
      'content' => schema_text,
      'content_length' => schema_text.length,
      'content_preview' => if session.task_params['preview_only']
                             schema_text[0..1000] + ((schema_text.length > 1000) ? "\n\n... (truncated, #{schema_text.length - 1000} more characters)" : '')
                           else
                             nil
                           end
                          })
  rescue => e
    session.logger.error("Failed to retrieve schema: #{e.message}")

    # Re-raise with context for Bolt to handle
    raise "Failed to retrieve schema '#{identifier}': #{e.message}"
  end
end

# Output the result as JSON for Bolt
puts result.to_json
