#!/usr/bin/env ruby
# frozen_string_literal: true

# Example task that lists NETCONF capabilities of a device
# This demonstrates basic connection and capability discovery
#
# Usage:
#   bolt task run list_capabilities --targets netconf://device.example.com

require 'json'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  session.logger.info('Listing NETCONF capabilities')

  # Get capabilities from the session
  capabilities = session.server_capabilities

  session.logger.info("Found #{capabilities.length} capabilities")

  # Format capabilities for display
  formatted_capabilities = capabilities.map.with_index do |cap, index|
    # Extract capability name and version/parameters
    if cap =~ %r{^(.*?)\?(.*)$}
      name = Regexp.last_match(1)
      params = Regexp.last_match(2).split('&').map { |p| "  #{p}" }.join("\n")
      "#{index + 1}. #{name}\n#{params}"
    else
      "#{index + 1}. #{cap}"
    end
  end

  # Use report_result to properly set the result in the report
  session.report_result({
                          'capabilities' => capabilities,
    'formatted' => formatted_capabilities.join("\n\n"),
    'count' => capabilities.length,
    'session_id' => session.session_id,
    'supports' => {
      'candidate' => session.supports_candidate?,
      'confirmed_commit' => session.supports_confirmed_commit?,
      'validate' => session.supports_validate?,
      'startup' => session.supports_startup?
    }
                        })
end

# Output the result as JSON for Bolt
puts result.to_json
