#!/usr/bin/env ruby
# frozen_string_literal: true

require 'json'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

# List all YANG schemas available on the device
#
# This task is a thin wrapper around Session#list_schemas which implements
# the standard NETCONF schema discovery defined in RFC 6022.
#
# Parameters:
#   - retrieve_sample: (optional) If true, retrieves a sample schema for demonstration

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Get schemas using the Session method
  schemas = session.list_schemas

  # Sort schemas by identifier for consistent output
  schemas.sort_by! { |s| s['identifier'] || '' }

  # Group schemas by namespace for better organization
  schemas_by_namespace = {}
  schemas.each do |schema|
    namespace = schema['namespace'] || 'unknown'
    schemas_by_namespace[namespace] ||= []
    schemas_by_namespace[namespace] << schema
  end

  # Create a summary
  summary = {
    'total_schemas' => schemas.length,
    'namespaces' => schemas_by_namespace.keys.count,
    'formats' => schemas.map { |s| s['format'] }.uniq.compact,
    'yang_modules' => schemas.count { |s| s['format'] == 'yang' },
    'yin_modules' => schemas.count { |s| s['format'] == 'yin' }
  }

  # Retrieve sample schema if requested
  sample_schema = nil
  if session.task_params['retrieve_sample'] && schemas.first
    begin
      schema = schemas.first
      schema_content = session.get_schema(
        schema['identifier'],
        schema['version'],
        schema['format'] || 'yang',
      )

      # Extract the schema data
      if schema_content && !schema_content.empty?
        # The element IS the data element
        if schema_content.first.name == 'data'
          schema_text = schema_content.first.text
        else
          # Try to find data element
          data_elem = schema_content.first.elements['data'] || schema_content.first.elements['//data']
          schema_text = data_elem&.text
        end

        sample_schema = {
          'identifier' => schema['identifier'],
          'version' => schema['version'],
          'format' => schema['format'],
          'content_preview' => if schema_text
                                 schema_text[0..500] + ((schema_text.length > 500) ? '...' : '')
                               else
                                 nil
                               end,
          'content_length' => schema_text ? schema_text.length : 0
        }
      end
    rescue => e
      session.logger.warn("Failed to retrieve sample schema: #{e.message}")
      sample_schema = {
        'identifier' => schemas.first['identifier'],
        'error' => e.message
      }
    end
  end

  # Return the result as a hash - with_session will merge this with the session report
  {
    'summary' => summary,
    'schemas' => schemas,
    'schemas_by_namespace' => schemas_by_namespace,
    'sample_schema_retrieval' => sample_schema
  }
end

# Output the result as JSON for Bolt
puts result.to_json
