#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require 'open3'
require 'json'
require 'shellwords'

begin
  require_relative '../../ruby_task_helper/files/task_helper'
rescue LoadError
  # include location for unit tests
  require 'fixtures/modules/ruby_task_helper/files/task_helper'
end

# Run Command task
class RunCommand < TaskHelper
  def configure_callback_plugins(task_directory, user, stdout_callback)
    require 'etc'
    require 'fileutils'

    if user
      # Configure permissions so that specified user can access callback plugins
      username = Etc.getpwnam(user)
      File.chown(username.uid, username.gid, task_directory)
      File.chown(username.uid, username.gid, "#{task_directory}/playbook_runner")
      File.chown(username.uid, username.gid, "#{task_directory}/playbook_runner/files/")
      File.chown(username.uid, username.gid, "#{task_directory}/playbook_runner/files/#{stdout_callback}.py")
    end
    "#{task_directory}/playbook_runner/files/"
  end

  def task(command_executor_path: nil,
           module_name: 'ansible.builtin.command',
           module_args: nil,
           project: nil,
           host_pattern: 'all',
           inventory: nil,
           user: nil,
           stdout_callback: 'puppet',
           additional_arguments: nil,
           **_kwargs)

    # Build environment hash
    env = {}
    env['ANSIBLE_STDOUT_CALLBACK'] = stdout_callback
    env['ANSIBLE_LOAD_CALLBACK_PLUGINS'] = '1'
    if stdout_callback.start_with?('puppet')
      env['ANSIBLE_CALLBACK_PLUGINS'] =
        configure_callback_plugins(ENV['PT__installdir'], user, stdout_callback)
    end

    # Build command array (NO SHELL)
    cmd = []
    cmd << (command_executor_path || 'ansible')
    cmd << host_pattern << '-m' << module_name
    cmd << '-a' << module_args if module_args
    cmd << '-i' << inventory if inventory
    cmd.concat(Shellwords.shellsplit("'#{additional_arguments}'")) if additional_arguments

    # If a user is provided, wrap the command in sudo *without a shell
    # Env vars have to be passed separately to ensure preserved with sudo
    if user
      sudo_cmd = ['sudo', '-u', user, 'env']
      env_array = env.map { |k, v| "#{k}=#{v}" }
      cmd = [sudo_cmd, env_array, cmd].flatten
      env = {}
    end

    # Execute safely, optionally in project directory
    if project
      stdout, stderr, status = Open3.capture3(env, *cmd, chdir: project)
    else
      stdout, stderr, status = Open3.capture3(env, *cmd)
    end

    # Display the output
    puts stdout

    # Raise error if running command has failed.
    return if status.success?

    raise TaskHelper::Error.new('Running command failed.',
                                'puppetlabs.tasks/command-error',
                                { cmd: cmd.join(' '), code: status.exitstatus, stderr: stderr })
  end
end

RunCommand.run if $PROGRAM_NAME == __FILE__
