require 'puppet'
require 'puppet/http'
require 'json'

module Puppet::Util::Gcp
  # Provides a class for interacting with GCP Postgres via the API
  class Client
    GCP_API_URL = 'sqladmin.googleapis.com'
    API_VERSION = 'v1'

    # Get the status information of the GCP Cloud SQL database
    #
    # @param [String] ca_cert file path to the CA cert for verifying GCP_API_URL
    # @param [String] token oauth token as a String with authorization to read from the project and db instance
    def initialize(ca_cert, token)
      @ca_cert = ca_cert
      @token = token
    end

    # Get the status information of the GCP Cloud SQL database
    #
    # @param [String] project_id String identifying the GCP project of the instance
    # @param [String] instance_id String identifying the instance
    #
    # @return [Hash]
    def get_pg_instance_info(project_id, instance_id)
      response = make_request(:get, "projects/#{project_id}/instances/#{instance_id}")

      JSON.parse(response.body)
    end

    # @return [Puppet::HTTP::Client]
    def get_http_client
      if Puppet.runtime.instance_variable_get(:@runtime_services).keys.include? :http
        runtime_service = :http
      else
        runtime_service = 'http'
      end
      client = Puppet.runtime[runtime_service]
    end

    # Create an SSL context with the ca certificate provided to the object when initialized
    #
    # This allows the HTTP client, which normally authenticates using the
    # Puppet infra certifcates, to authenticate with the GCP servers
    #
    # return [Puppet::SSL::SSLContext]
    def get_ssl_context()
      ssl_provider = Puppet::SSL::SSLProvider.new
      google_root = [OpenSSL::X509::Certificate.new(File.read(@ca_cert))]
      ssl_provider.create_root_context(cacerts: google_root, crls: [], revocation: false)
    end

    def make_request(type, endpoint)
      client = get_http_client
      headers = {'Content-Type' => 'application/json; charset=utf-8',
                 'Authorization' => "Bearer #{@token}"}
      options = { ssl_context: get_ssl_context }

      full_uri = URI("https://#{GCP_API_URL}/#{API_VERSION}/#{endpoint}")
      case type
      when :get
        response = client.get(full_uri, headers: headers, options: options)
      else
        raise Puppet::Error, "GCP client make_request called with invalid request type #{type}"
      end

      case response.code
      when 200..399
        return response
      else
        raise Puppet::Error, "Received an unexpected error response from #{full_uri}: #{response.code} #{response.body}"
      end
    end
  end
end
