require 'puppet/application/face_base'

# This subcommand is implemented as a face. The definition of the application
# can be found in face/infrastructure.rb.
class Puppet::Application::Infrastructure < Puppet::Application::FaceBase
  # Call Puppet's settings.use method (which is idempotent) during the
  # application's setup phase to have it create all necessary objects and
  # directories for a PE install via a puppet apply.
  #
  # This is required in order for the `infrastructure configure` command to run
  # before a puppet agent run.  Otherwise directories may not exist, such as
  # statedir (where agent lockfile is kept).
  def setup
    super
    Puppet.settings.use(:main)
  end

  # A copy-pasted implementation of FaceBase's main method with only the
  # Puppet.err call at the end instructing users how to call `help` changed.
  def main
    status = false

    # Call the method associated with the provided action (e.g., 'find').
    unless @action
      puts Puppet::Face[:help, :current].help(@face.name)
      raise _("%{face} does not respond to action %{action}") % {face: face, action: arguments.first}
    end

    # We need to do arity checking here because this is generic code
    # calling generic methods – that have argument defaulting.  We need to
    # make sure we don't accidentally pass the options as the first
    # argument to a method that takes one argument.  eg:
    #
    #   puppet facts find
    #   => options => {}
    #      @arguments => [{}]
    #   => @face.send :bar, {}
    #
    #   def face.bar(argument, options = {})
    #   => bar({}, {})  # oops!  we thought the options were the
    #                   # positional argument!!
    #
    # We could also fix this by making it mandatory to pass the options on
    # every call, but that would make the Ruby API much more annoying to
    # work with; having the defaulting is a much nicer convention to have.
    #
    # We could also pass the arguments implicitly, by having a magic
    # 'options' method that was visible in the scope of the action, which
    # returned the right stuff.
    #
    # That sounds attractive, but adds complications to all sorts of
    # things, especially when you think about how to pass options when you
    # are writing Ruby code that calls multiple faces.  Especially if
    # faces are involved in that. ;)
    #
    # --daniel 2011-04-27
    if (arity = @action.positional_arg_count) > 0
      unless (count = arguments.length) == arity then
        num_expected_args = arity - 1
        num_given_args = count - 1
        plural_expected_args = n_("argument", "arguments", num_expected_args)
        plural_given_args = n_("argument", "arguments", num_given_args)
        raise ArgumentError, _("puppet %{face} %{action} takes %{num_expected_args} %{plural_expected_args}, but you gave %{num_given_args} %{plural_given_args}") % { face: @face.name, action: @action.name, num_expected_args: num_expected_args, plural_expected_args: plural_expected_args, num_given_args: num_given_args, plural_given_args: plural_given_args }
      end
    end

    result = @face.send(@action.name, *arguments)
    puts render(result, arguments) unless result.nil?
    status = true

  # We need an easy way for the action to set a specific exit code, so we
  # rescue SystemExit here; This allows each action to set the desired exit
  # code by simply calling Kernel::exit.  eg:
  #
  #   exit(2)
  #
  # --kelsey 2012-02-14
  rescue SystemExit => detail
    status = detail.status

  rescue => detail
    Puppet.log_exception(detail)
    case detail
    when PuppetX::Util::OrchestratorError
      # Directing to 'help' usage is not helpful for errors returned from orchestrator jobs
    else
      Puppet.err _("Try 'puppet infrastructure help %{action}' for usage") % { action: @action.name }
    end

  ensure
    exit status
  end
end
