require 'puppet_x/puppetlabs/meep/caching'
require 'puppet_x/puppetlabs/meep/infra/defaults'
require 'puppet_x/puppetlabs/meep/hiera_adapter'

module PuppetX
module Puppetlabs
module Meep
module Infra

  # This is a cluster of utilities for looking up system information from
  # puppet resource faces, from puppet-lookup (hiera data), or from specific
  # facts.
  module Lookup
    include PuppetX::Puppetlabs::Meep::Caching
    include PuppetX::Puppetlabs::Meep::Infra::Defaults

    def enterprise_hiera_yaml
      '/etc/puppetlabs/enterprise/hiera.yaml'
    end

    # A handle for puppet lookup set to the enterprise hiera configuration.
    # This let's us interogate pe.conf as the current node.
    def enterprise_lookup_handle
      cached(:enterprise_lookup_handle) do
        PuppetX::Puppetlabs::Meep::HieraAdapter.new(enterprise_hiera_yaml)
      end
    end

    # Lookup a parameter in the Hiera hierarchy pointed to by the current setting
    # of Puppet[:hiera_config]. This is intended to lookup current parameter
    # values set in pe.conf, but this function must be called within the context
    # of a puppet_infrastructure_context() block in order for hiera_config to have
    # been set properly.
    #
    # @param parameter [String] the parameter key to lookup.
    # @return [String] the found value or nil if the parameter was not found.
    def puppet_lookup(parameter)
      node = cached(:puppet_lookup_node) do
        PuppetX::Puppetlabs::Meep::HieraAdapter.get_node
      end
      scope = cached(:puppet_lookup_scope) do
        PuppetX::Puppetlabs::Meep::HieraAdapter.generate_scope(node)
      end
      enterprise_lookup_handle.lookup(parameter, scope)
    end

    def package_find(id)
      res = nil
      os_package_providers = [:apt, :yum, :dnf, :zypper] # :dpkg, :rpm
      package_type = Puppet::Type.type(:package)
      package_type.providers_by_source.each do |provider|
        next unless os_package_providers.include? provider.name
        begin
          provider.instances.each do |resource_instance|
            props = resource_instance.properties
            res = props if props[:name] == id

            # If the res hash doesn't exist, create new Package resource for the nonexistent package
            res ||= package_type.new(:name => id, :audit => package_type.properties.collect { |s| s.name }).to_resource.to_hash
          end
        rescue => detail
          Puppet.log_exception(detail, "Cannot collect packages for #{provider} provider; #{detail}")
        end
      end
      res
    end

    # Retrieve resource information from a particular puppet type on the local node.
    #
    # @param resource [String] the provider name ('package', 'user', etc.)
    # @param id [String] the unique name of the resource to lookup.
    # @return [Hash] Returns the Hash of data about the resource from the provider.
    def resource_lookup(resource, id)
      resource_face = Puppet::Face[:resource, :current]
      if resource == 'package'
        package_find(id)
      else
        resource_face.find("#{resource}/#{id}")
      end
    end

    def package_version(package_name)
      package_resource = resource_lookup('package', package_name)
      package_resource[:ensure]
    end

    def package_present?(package_name)
      case package_version(package_name)
      when :purged, :absent then false
      else true
      end
    end

    def fqdn
      cached(:fqdn) do
        Facter.value('fqdn')
      end
    end

    def pe_server_version
      cached(:pe_server_version) do
        Facter.value('pe_server_version')
      end
    end

    def pe_postgresql_info
      cached(:pe_postgresql_info) do
        Facter.value('pe_postgresql_info') || {}
      end
    end

    def fips_enabled?
      cached(:fips_enabled) do
        Facter.value('fips_enabled') || false
      end
    end

    # The default for PE's current postgres version is dependent on whether
    # FIPS is enabled.
    def platform_current_postgres_version
      fips_enabled? ?
        infra_default(:CURRENT_FIPS_POSTGRES_VERSION) :
        infra_default(:CURRENT_POSTGRES_VERSION)
    end

    # @return the major version of the currently installed pe-postgresql-server
    #   package or nil if not installed.
    def postgres_server_version
      version = pe_postgresql_info['installed_server_version']
      version == '' ? nil : version
    end

    def pe_postgres_uid
      pe_postgres_user = resource_lookup('user', 'pe-postgres')
      pe_postgres_user[:uid] if pe_postgres_user[:ensure] == :present
    end

    def password_omitted?
      console_admin_password = puppet_lookup("console_admin_password")

      # At the time of this change, console_admin_password can not be empty
      # The empty check is in place to prevent future changes that may allow it to be empty from
      # creating a bug.
      console_admin_password.nil? || console_admin_password.empty?
    end

    # Allows specifying a different postgres version to be considered for
    # migration during install by specifying the
    # puppet_enterprise::postgres_version_override parameter in pe.conf.
    #
    # This is useful to test upgrading to a newer version of pe-postgresql
    # before we have bumped the actual defaults that would force the behavior.
    # This way we can test the upgrades prior to breaking all of ci.
    #
    # @return [String] Either the puppet_enterprise::postgres_version_override
    #   value if available from a puppet_lookup() or the platform specific default.
    def requested_postgres_version
      puppet_lookup('puppet_enterprise::postgres_version_override') || platform_current_postgres_version
    end

    # Determines the previous postgres version based on the value of requested_postgres_version.
    # This allows us to test migrations to new postgres versions provided
    # via the puppet_enterprise::postgres_version_override parameter.
    #
    # @return [String] The Postgresql version used prior to the version we are 
    #   installing/upgrading to.
    def previous_postgres_version
      seq = infra_default(:POSTGRES_VERSION_SEQUENCE)
      ind = seq.index(requested_postgres_version)
      if ind.nil? || ind == 0
        Puppet.log_exception("Invalid current postgres version detected. Ensure that puppet_enterprise::postgres_version_override is set to one of #{seq[1..-1]}")
      end
      seq[ind-1]
    end
  end
end
end
end
end
