require 'pp'

module PuppetX
  module Util
    class OrchestratorError < StandardError
    end

    class OrchestratorJobFailedError < OrchestratorError
      attr_reader :message, :details

      def initialize(message, details)
        msg = _('An error has occurred while running an orchestrated job. (%{message})') % { message: message }
        if details.empty?
          msg += ' ' + _('No node details were retrieved.')
        else
          details.each do |d|
            name = d['name']
            report_url = d['report-url']
            detail = d.reject { |k, _| ['name', 'report-url'].include?(k) }
            msg += report_url.nil? ?
              "\n    " + _('%{name}: (%{detail})') % { name: name, report_url: report_url, detail: detail } :
              "\n    " + _('%{name}: See the report at %{report_url} for more details. (%{detail})') % { name: name, report_url: report_url, detail: detail }
          end
        end
        super(msg)
        @message = msg
        @details = details
      end
    end

    class OrchestratorTaskFailedError < OrchestratorError
      attr_reader :msg, :kind, :details

      def type
        'task'
      end

      def initialize(msg, kind, details)
        err_msg_array = [
          _("An error has occurred while running orchestrated %{type}.") % { type: type },
          _("\tMessage: %{msg}\n\tKind: %{kind}\n\tDetails:") % { msg: msg, kind: kind },
        ]
        case details
        when Hash
          details.each do |k,v|
            err_msg_array << "\t\t#{k}: #{v.kind_of?(::String) ? v : v.pretty_inspect}"
          end
        else
          err_msg_array << details.pretty_inspect
        end
        super(err_msg_array.join("\n"))
        @msg = msg
        @kind = kind
        @details = details
      end
    end

    class OrchestratorPlanFailedError < OrchestratorTaskFailedError
      def type
        'plan'
      end
    end

    class OrchestratorJobStoppedError < OrchestratorError
      attr_reader :job_id

      def initialize(job_id)
        super(_("An error has occurred while running orchestrated job.") + " " +
          _("The orchestrator job %{job_id} was externally stopped.") % { job_id: job_id })
        @job_id = job_id
      end
    end

    class OrchestratorJobTimeoutError < OrchestratorError
      attr_reader :timeout, :job_id

      def initialize(timeout, job_id)
        super(_("An error has occurred while running orchestrated job.") + " " +
          _("The orchestration service returned an error.") + " " +
          _("Verify that the service is running and has no relevant alerts using 'puppet infra status', then check the status of the job with 'puppet job show %{job_id}'.") % { job_id: job_id })
        @timeout = timeout
        @job_id = job_id
      end
    end

    class OrchestratorEmptyResponseError < OrchestratorError
      def initialize()
        super(_("An error has occurred while running orchestrated job.") + " " +
          _("The orchestration service returned an empty response.") + " " +
          _("See the orchestration services log file for more details."))
      end
    end

    class OrchestratorResponseError < OrchestratorError
      attr_reader :status_code, :body

      def initialize(status_code, body)
        super(_("An error has occurred while running orchestrated job.") + " " +
          _("The orchestration service returned an error response.") + " " +
          _("See the orchestration services log file for more details.") + " " +
          "(" + _("status") + " %{status_code}: %{body}" % { status_code: status_code, body: body })
        @status_code = status_code
        @body = body
      end
    end

    class OrchestratorEmptyTargetError < OrchestratorError
      def initialize(scope)
        message = [
          _("An error has occurred while running orchestrated job."),
          _("The orchestration service unexpectedly raised a puppetlabs.orchestrator/empty-target error for the given scope: '%{scope}'.") % { scope: scope },
          _("See the orchestration services log file for more details."),
        ].join(' ')
        super(message)
      end
    end
  end
end
