require 'open3'
require 'puppet_x/util/stringformatter'

module PuppetX
  module Util
    class Shell

      # The difference between Puppet::Util::Log.levels and what destinations prints...
      LOG_ABBREVIATIONS = {
        critical: :crit,
        emergency: :emerg,
        error: :err,
      }.freeze

      # Execute command via Open3.popen2e logging both stdout and stderr lines
      # via Puppet logging facility.
      #
      # @return [Process::Status]
      def self.run_and_log(command)
        Open3.popen2e(command) do |_, output, wait_thread|
          output.each do |line|
            bare = PuppetX::Util::String::Formatter.uncolorize(line.chomp)
            level = nil
            if matcher = (bare).match(%r{\A\[?(\w+)\]?:\s*(.+)})
              level = matcher[1].downcase.to_sym
              level = LOG_ABBREVIATIONS[level] || level
              message = Puppet::Util::Log.levels.include?(level) ? matcher[2] : bare
            else
              message = bare
            end
            case level
            when :emerge, :alert, :crit, :err, :warning
              Puppet.send(level, message)
            when :info,:debug
              if Puppet::Util::Log.sendlevel?(level)
                Puppet.send(level, message)
              else
                Puppet.notice(bare)
              end
            else
              Puppet.notice(message)
            end
          end
          wait_thread.value
        end
      end
    end
  end
end
