# Ciphers before TLS 1.3 have an IANA/JSSE format, and an OpenSSL format.
# Jetty/Java requires the former, orch/bolt-server/ace-server/nginx/postgres the latter.
# Translations between the two can be found at
# https://testssl.sh/openssl-iana.mapping.html
# https://www.openssl.org/docs/man1.1.1/man1/ciphers.html
#
# Cipher lists were derived from https://confluence.puppetlabs.com/display/SRE/TLS+Usage+Standard
# which is derived from the Mozilla Intermediate list. This was updated 1/2024.
# https://perforce.atlassian.net/browse/PE-36931
#
# -----------------------------------------------------------------------------
#  IMPORTANT DETAILS FOR USAGE IN PUPPET ENTERPRISE
#  This function can be used as the authoritative list of cipher defaults used
#  in Puppet Enterprise with the following caveats.
# -----------------------------------------------------------------------------
# - Because puppet/puppetserver only generates RSA keys, we effectively only use the RSA ciphers here.
#   We might explore ECC keys to support ECDSA in the future.  However, we include a couple ECDSA ciphers
#   in case a user wants to generate their own certs/keys. Be aware, though, that these suites won't be useable
#   by a regular PE install or in the browser with the PE console using puppet/puppetserver-generated certs/keys.
# - FIPS_allowed is the list of ciphers allowed on FIPS installs. This is dervied by collecting all the ciphers we
#   use in PE, and then filtering based on https://www.gsa.gov/cdnstatic/SSL_TLS_Implementation_%5BCIO_IT_Security_14-69_Rev_4%5D_5-26-2020docx.pdf
# - TLS_CHACHA20_POLY1305_SHA256 cipher is allowed in 1.3 non-FIPS installs. It is not FIPS compliant as of now.
#   https://csrc.nist.gov/CSRC/media/Publications/fips/140/2/final/documents/fips1402annexa.pdf
# - TLSv1 and TLSv1.1 lists are the same. Copied here so one or both can be passed in to the function.
# - Puma-based services are unable to limit the SSL protocol used, so regardless of the cipher
#   list provided or the puppet_enterprise::ssl_protocols setting, it will allow all TLS versions.
#   This may change in the future to disallow TLSv1 and TLSv1.1.
# - PostgreSQL 14 can limit the SSL protocols used. However, there is
#   no setting in postgresql.conf for controlling 1.3 ciphers, only 1.2 or less.

function puppet_enterprise::ciphers(
  Array[Enum['TLSv1.3', 'TLSv1.2', 'TLSv1.1', 'TLSv1']] $cipher_groups,
  Enum['openssl', 'iana'] $format,
) >> Array[String] {
        
  # IANA => OpenSSL
  # When adding a cipher, make sure you add the OpenSSL translation here too.
  $translation = {
    'TLS_AES_256_GCM_SHA384'                        => 'TLS_AES_256_GCM_SHA384',
    'TLS_CHACHA20_POLY1305_SHA256'                  => 'TLS_CHACHA20_POLY1305_SHA256',
    'TLS_AES_128_GCM_SHA256'                        => 'TLS_AES_128_GCM_SHA256',
    'TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384'         => 'ECDHE-RSA-AES256-GCM-SHA384',
    'TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384'       => 'ECDHE-ECDSA-AES256-GCM-SHA384',
    'TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256'   => 'ECDHE-RSA-CHACHA20-POLY1305',
    'TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256' => 'ECDHE-ECDSA-CHACHA20-POLY1305',
    'TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256'         => 'ECDHE-RSA-AES128-GCM-SHA256',
    'TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256'       => 'ECDHE-ECDSA-AES128-GCM-SHA256',
    'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA'            => 'ECDHE-RSA-AES256-SHA',
    'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA'          => 'ECDHE-ECDSA-AES256-SHA',
    'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA'            => 'ECDHE-RSA-AES128-SHA',
    'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA'          => 'ECDHE-ECDSA-AES128-SHA',
    'TLS_DHE_RSA_WITH_AES_256_CBC_SHA'              => 'DHE-RSA-AES256-SHA',
    'TLS_DHE_RSA_WITH_AES_128_CBC_SHA'              => 'DHE-RSA-AES128-SHA',
  }

  # When updating these lists, use the IANA format.
  $ciphers = {
    'TLSv1.3' => [
      'TLS_AES_256_GCM_SHA384',
      'TLS_CHACHA20_POLY1305_SHA256',
      'TLS_AES_128_GCM_SHA256',
    ],
    'TLSv1.2' => [
      'TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384',
      'TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384',
      'TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256',
      'TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256',
      'TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256',
      'TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256',
    ],
    'TLSv1.1' => [
      'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
      'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
      'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
      'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
      'TLS_DHE_RSA_WITH_AES_256_CBC_SHA',
      'TLS_DHE_RSA_WITH_AES_128_CBC_SHA',
    ],
    'TLSv1' => [
      'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
      'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
      'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
      'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
      'TLS_DHE_RSA_WITH_AES_256_CBC_SHA',
      'TLS_DHE_RSA_WITH_AES_128_CBC_SHA',
    ],
    'FIPS_allowed' => [
      'TLS_AES_256_GCM_SHA384',
      'TLS_AES_128_GCM_SHA256',
      'TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384',
      'TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384',
      'TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256',
      'TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256',
    ],
  }

  # Probably shouldn't be able to pass in an empty protocol list,
  # but just in case.
  $_cipher_groups = empty($cipher_groups) ? {
    true => ['TLSv1.3', 'TLSv1.2'],
    default => $cipher_groups,
  }

  $_cipher_list = $_cipher_groups.map |$group| { $ciphers[$group] }.flatten.unique
  $_filtered_cipher_list = $facts['fips_enabled'] ? {
    true => $_cipher_list.filter |$c| { $c in $ciphers['FIPS_allowed'] },
    default => $_cipher_list,
  }

  $cipher_list = ($format == 'openssl') ? {
    true => $_filtered_cipher_list.map |$c| { $translation[$c] },
    default => $_filtered_cipher_list,
  }
}
