class puppet_enterprise::master::code_manager(
  String $certname                              = $facts['clientcert'],
  String $webserver_ssl_host                    = $puppet_enterprise::params::ssl_address,
  Integer $webserver_ssl_port                   = 8170,
  Array[String] $ssl_protocols                  = $puppet_enterprise::ssl_protocols,
  Array[String] $ssl_cipher_suites              = $puppet_enterprise::ssl_cipher_suites,
  String $hostcrl                               = $puppet_enterprise::params::hostcrl,
  String $localcacert                           = $puppet_enterprise::params::localcacert,
  String $datadir                               = '/opt/puppetlabs/server/data/code-manager/',
  String $cachedir                              = '/opt/puppetlabs/server/data/code-manager/cache',
  Optional[Boolean] $file_sync_auto_commit      = undef,
  Optional[String] $file_sync_repo_id           = undef,
  Boolean $authenticate_webhook                 = true,
  Boolean $full_deploy                          = false,
  Integer $deploy_pool_size                     = 2,
  Integer $deploy_pool_cleanup_interval         = 100,
  Integer $download_pool_size                   = 4,
  Boolean $r10k_trace                           = false,
  Integer $timeouts_fetch                       = 30,
  Integer $timeouts_deploy                      = 600,
  Integer $timeouts_hook                        = 30,
  Integer $timeouts_shutdown                    = 610,
  Integer $timeouts_wait                        = 700,
  Integer $timeouts_sync                        = 300,
  Optional[Array[String]] $purge_whitelist      = undef,
  Optional[Array[String]] $purge_allowlist      = pe_pick($purge_whitelist, ['.resource_types']),
  Optional[String] $environmentdir              = undef,
  Optional[Hash] $forge_settings                = undef,
  Optional[Hash] $git_settings                  = undef,
  Optional[Hash] $git_settings_from_code_mgmt   = undef,
  Optional[Hash] $forge_settings_from_code_mgmt = undef,
  Optional[Hash] $sources                       = undef,
  Optional[String] $invalid_branches            = undef,
  Optional[String] $remote                      = undef,
  Optional[String] $proxy                       = undef,
  Optional[String] $private_key                 = undef,
  Optional[Array[Hash]] $post_environment_hooks = undef,
  Integer $puppet_master_port                   = $puppet_enterprise::puppet_master_port,
  Optional[Enum['rugged', 'shellgit']] $git_provider = undef,
) inherits puppet_enterprise::params {

  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::master::code_manager::purge_whitelist': replacement => 'puppet_enterprise::master::code_manager::purge_allowlist' }

  $confdir = '/etc/puppetlabs/puppetserver'

  if ($file_sync_repo_id != undef) {
    warning('Deprecation: $puppet_enterprise::master::code_manager::file_sync_repo_id is deprecated and will be ignored')
  }

  if ($file_sync_auto_commit != undef) {
    warning('Deprecation: $puppet_enterprise::master::code_manager::file_sync_auto_commit is deprecated and will be ignored')
  }

  if ($git_settings != undef) {
    warning('Deprecation: $puppet_enterprise::master::code_manager::git_settings is deprecated, please use explicit `git_*` values (e.g., git_oauth_token) on the puppet_enterprise::master::code_management class.')
  }

  if ($forge_settings != undef) {
    warning('Deprecation: $puppet_enterprise::master::code_manager::forge_settings is deprecated, please use explicit `forge_*` values (e.g., forge_authorization_token) on the puppet_enterprise::master::code_management class.')
  }

  if ($private_key != undef) {
    warning('Deprecation: $puppet_enterprise::master::code_manager::private_key is deprecated, please use git_private_key on the puppet_enterprise::master::code_management class.')
  }

  $git_configuration = [
    [ 'private-key', $private_key],
    [ 'provider', $git_provider]
  ]
  $configured_git_values = $git_configuration.filter |$pair| { $pair[1] }
  $git_config = Hash($configured_git_values)

  $code_manager_service = 'code-manager-v1'
  if !pe_empty($environmentdir) {
    $_environmentdir = $environmentdir
  }
  else {
      $_environmentdir = '/etc/puppetlabs/code-staging/environments'
  }

  $web_router_section = "web-router-service.\"puppetlabs.code-manager.services/${code_manager_service}\""

  if !pe_empty($sources) {
      pe_validate_hash($sources)
      $_sources = $sources
  } elsif !pe_empty($remote) {
    $_sources = {
      'puppet' => {
        'remote' => $remote,
      },
    }
  } else {
    $_sources = undef
  }

  Pe_hocon_setting {
    ensure => present,
    notify => Service['pe-puppetserver'],
  }

  File {
    ensure  => present,
    owner   => 'pe-puppet',
    group   => 'pe-puppet',
    mode    => '0640',
    require => Package['pe-puppetserver'],
  }

  file { $datadir :
    ensure => directory,
    mode   => '0640',
  }

  file { "${confdir}/conf.d/code-manager.conf" :
    ensure => present,
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { $code_manager_service :
    container => 'puppetserver',
    namespace => 'puppetlabs.code-manager.services',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { 'remote-activity-reporter' :
    container => 'puppetserver',
    namespace => 'puppetlabs.rbac-client.services.activity',
  }

  puppet_enterprise::trapperkeeper::webserver_settings { 'code-manager' :
    container          => 'puppetserver',
    certname           => $certname,
    ssl_listen_address => $webserver_ssl_host,
    ssl_listen_port    => $webserver_ssl_port,
    localcacert        => $localcacert,
    hostcrl            => $hostcrl,
    access_log_config  => "${confdir}/code-manager-request-logging.xml",
    ssl_protocols      => $ssl_protocols,
    ssl_cipher_suites  => $ssl_cipher_suites,
  }

  pe_hocon_setting { "${web_router_section}.route":
    path    => "${confdir}/conf.d/web-routes.conf",
    setting => "${web_router_section}.route",
    value   => '/code-manager/v1',
  }
  pe_hocon_setting { "${web_router_section}.server":
    path    => "${confdir}/conf.d/web-routes.conf",
    setting => "${web_router_section}.server",
    value   => 'code-manager',
  }

  pe_hocon_setting{ 'code-manager.file-sync.repo-id':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.file-sync.repo-id',
    value   => 'puppet-code',
  }
  pe_hocon_setting{ 'code-manager.environments.environmentdir':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.environments.environmentdir',
    value   => $_environmentdir,
  }
  # This setting is deprecated in favor of code-manager.environments.environmentdir
  pe_hocon_setting{ 'code-manager.environmentdir':
    ensure  => absent,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.environmentdir',
  }
  pe_hocon_setting{ 'code-manager.datadir':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.datadir',
    value   => $datadir,
  }
  pe_hocon_setting{ 'code-manager.cachedir':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.cachedir',
    value   => $cachedir,
  }
  pe_hocon_setting{ 'code-manager.authenticate-webhook':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.authenticate-webhook',
    value   => $authenticate_webhook,
  }
  pe_hocon_setting{ 'code-manager.full-deploy':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.full-deploy',
    value   => $full_deploy,
  }
  pe_hocon_setting{ 'code-manager.deploy-pool-size':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.deploy-pool-size',
    value   => $deploy_pool_size,
  }
  pe_hocon_setting{ 'code-manager.deploy-pool-cleanup-interval':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.deploy-pool-cleanup-interval',
    value   => $deploy_pool_cleanup_interval,
  }
  pe_hocon_setting{ 'code-manager.download-pool-size':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.download-pool-size',
    value   => $download_pool_size,
  }
  pe_hocon_setting{ 'code-manager.r10k-trace':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.r10k-trace',
    value   => $r10k_trace,
  }
  pe_hocon_setting{ 'code-manager.timeouts.fetch':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.timeouts.fetch',
    value   => $timeouts_fetch,
  }
  pe_hocon_setting{ 'code-manager.timeouts.deploy':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.timeouts.deploy',
    value   => $timeouts_deploy,
  }
  pe_hocon_setting{ 'code-manager.timeouts.hook':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.timeouts.hook',
    value   => $timeouts_hook,
  }
  pe_hocon_setting{ 'code-manager.timeouts.shutdown':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.timeouts.shutdown',
    value   => $timeouts_shutdown,
  }
  pe_hocon_setting{ 'code-manager.timeouts.wait':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.timeouts.wait',
    value   => $timeouts_wait,
  }
  pe_hocon_setting{ 'code-manager.timeouts.sync':
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.timeouts.sync',
    value   => $timeouts_sync,
  }

  if pe_empty($purge_allowlist) {
    $purge_allowlist_ensure = 'absent'
  } else {
    $purge_allowlist_ensure = 'present'
  }

  pe_hocon_setting { 'code-manager.purge-allowlist':
    ensure  => $purge_allowlist_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.purge-allowlist',
    type    => 'array',
    value   => $purge_allowlist,
  }

  pe_hocon_setting { 'code-manager.purge-whitelist':
    ensure  => absent,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.purge-whitelist',
    type    => 'array',
  }

  if pe_empty($proxy) {
    $proxy_ensure = absent
  } else {
    $proxy_ensure = present
  }

  # This is the proxy for all r10k operations and can be selectively overridden
  # for either git or forge operations by their respective *_settings params.
  pe_hocon_setting { 'code-manager.proxy':
    ensure  => $proxy_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.proxy',
    value   => $proxy,
  }

  if pe_empty($post_environment_hooks) {
    $post_environment_hooks_ensure = absent
  } else {
    $post_environment_hooks_ensure = present
  }
  pe_hocon_setting { 'code-manager.hooks.post-environment':
    ensure  => $post_environment_hooks_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.hooks.post-environment',
    type    => 'array',
    value   => $post_environment_hooks,
  }

  if !pe_empty($forge_settings) and !pe_empty($forge_settings_from_code_mgmt) {
    $forge_settings.each |$pair| {
      $k = $pair[0]
      $v = $pair[1]
      if $k in $forge_settings_from_code_mgmt {
        $p = "forge_${regsubst($k, '-', '_', 'G')}"
        $message = @("EOT"/L)
        '${k}' is configured both via the explicit param 'puppet_enterprise::master::code_management::${p}' \
        and via 'forge_settings' which includes { '${k}' => '${forge_settings[k]}' }. \
        We will use the value of '${v}' from the code_management parameter '${p}'.
        | - EOT
        notify { "Warn forge_settings $k override from explicit param":
          message => $message,
        }
      }
    }
  }

  # Ensure these are actual Hashes we can merge and not Undef values
  if $forge_settings {
    $_forge_settings = $forge_settings
  } else {
    $_forge_settings = Hash([])
  }

  if $forge_settings_from_code_mgmt {
    $_forge_settings_from_code_mgmt = $forge_settings_from_code_mgmt
  } else {
    $_forge_settings_from_code_mgmt = Hash([])
  }

  $_forge = $_forge_settings + $_forge_settings_from_code_mgmt

  if pe_empty($_forge) {
    $forge_ensure = absent
  } else {
    $forge_ensure = present
  }

  pe_hocon_setting{ 'code-manager.forge':
    ensure  => $forge_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.forge',
    value   => $_forge,
  }
  # PE 2015.3.0 and 2015.3.1 set this invalid setting.
  pe_hocon_setting{ 'code-manager.forge-settings':
    ensure  => absent,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.forge-settings',
  }

  if pe_empty($_sources) {
    $sources_ensure = absent
  } else {
    $sources_ensure = present
  }
  pe_hocon_setting{ 'code-manager.environments.sources':
    ensure  => $sources_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.environments.sources',
    value   => $_sources,
  }
  # This setting is deprecated in favor of code-manager.environments.sources
  pe_hocon_setting{ 'code-manager.sources':
    ensure  => absent,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.sources',
  }

  if pe_empty($invalid_branches) {
    $invalid_branches_ensure = absent
  } else {
    $invalid_branches_ensure = present
  }

  pe_hocon_setting{ 'code-manager.invalid-branches':
    ensure  => $invalid_branches_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.invalid-branches',
    value   => $invalid_branches,
  }

  if !pe_empty($git_settings) and !pe_empty($git_settings_from_code_mgmt) {
    $git_settings.each |$pair| {
      $k = $pair[0]
      $v = $pair[1]
      if $k in $git_settings_from_code_mgmt {
        $p = "git_${regsubst($k, '-', '_', 'G')}"
        $message = @("EOT"/L)
          '${k}' is configured both via the explicit param 'puppet_enterprise::master::code_management::${p}' \
          and via 'git_settings'. The value of '${p}' is '${git_settings_from_code_mgmt[$k]}', \
          and 'git_settings' includes { '${k}' => '${v}' }. \
          We will use the value of '${v}' from the code_management parameter '${p}'.
          | - EOT
        notify { "Warn git_settings $k override from explicit param":
          message => $message,
        }
      }
    }
  }

  if !pe_empty($git_settings) {
    if $git_settings['private-key'] and $private_key {
      fail('Both private_key and git_settings provided to puppet_enterprise::master::code-manager.')
    }

    if $git_settings['provider'] and $git_provider {
      $message = @("EOT"/L)
        The git provider is configured both via the explicit param 'git_provider' \
        and via 'git_settings'. The value of 'git_provider' is '${git_provider}', \
        and 'git_settings' includes { 'provider' => '${git_settings['provider']}' }. \
        We will use the value of '${git_provider}' from the explicit parameter.
        | - EOT
      notify { "Warn git_provider override":
        message => $message,
      }
    }
  }

  # Ensure these are actual Hashes we can merge and not Undef values
  if $git_settings {
    $_git_settings = $git_settings
  } else {
    $_git_settings = Hash([])
  }

  if $git_settings_from_code_mgmt {
    $_git_settings_from_code_mgmt = $git_settings_from_code_mgmt
  } else {
    $_git_settings_from_code_mgmt = Hash([])
  }

  $_git = $_git_settings + $_git_settings_from_code_mgmt + $git_config

  if pe_empty($_git) {
    $git_ensure = absent
  } else {
    $git_ensure = present
  }

  # This will now almost always be set because we're including the git_provider here
  pe_hocon_setting{ 'code-manager.git':
    ensure  => $git_ensure,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.git',
    value   => $_git,
  }

  # We should go into code-manager and deprecate this setting
  unless pe_empty($git_provider) {
    pe_hocon_setting{ 'code-manager.git-provider':
      ensure  => 'present',
      path    => "${confdir}/conf.d/code-manager.conf",
      setting => 'code-manager.git-provider',
      value   => $git_provider,
    }
  }

  # This setting is deprecated
  pe_hocon_setting {'code-manager.private-key':
    ensure  => absent,
    path    => "${confdir}/conf.d/code-manager.conf",
    setting => 'code-manager.private-key',
  }

}
