require 'hocon'
require 'json'
require 'net/https'
require 'uri'
require 'openssl'
require 'puppet'
require 'benchmark'
require 'base64'
require 'pe_backup_tools/utils/command_summary'
require 'pe_backup_tools/utils/mixins'
require 'facter'
require 'open3'
require 'puppet/util/log'

module PeBackupTools
  # This module provides utility functions related to a variety
  # of things  (passwords, GET/POST, certificates) for our instance commands
  module Utils
    extend PeBackupTools::Utils::Mixins

    def self.get_pe_version
      # TODO: Get this from the Facter fact maybe?
      File.read('/opt/puppetlabs/server/pe_version').chomp
    end

    def self.console_cert_dir
      '/opt/puppetlabs/server/data/console-services/certs'
    end

    def self.get_cert_conf(certname, cert_dir = console_cert_dir)
      {
        'ca_cert' => '/etc/puppetlabs/puppet/ssl/certs/ca.pem',
        'private_key' => "#{cert_dir}/#{certname}.private_key.pem",
        'cert' => "#{cert_dir}/#{certname}.cert.pem"
      }
    end

    def self.get_agent_cert_conf(certname)
      {
        'ca_cert' => '/etc/puppetlabs/puppet/ssl/certs/ca.pem',
        'private_key' => "/etc/puppetlabs/puppet/ssl/private_keys/#{certname}.pem",
        'cert' => "/etc/puppetlabs/puppet/ssl/certs/#{certname}.pem"
      }
    end

    def self.build_auth(uri, cert_conf)
      https = Net::HTTP.new(uri.host, uri.port)
      https.use_ssl = true
      https.ca_file = cert_conf['ca_cert']
      https.key = OpenSSL::PKey::RSA.new(File.read(cert_conf['private_key']))
      https.cert = OpenSSL::X509::Certificate.new(File.read(cert_conf['cert']))
      https.verify_mode = OpenSSL::SSL::VERIFY_PEER
      https
    end

    def self.configure_logging(logfile, options)
      Puppet::Util::Log.newdestination(logfile)
      loglevel = options[:debug] ? :debug : :info
      Puppet::Util::Log.level = loglevel
    rescue Errno::EACCES => _e
      error(_('Unable to access log file %{logfile} to configure logging.') % { logfile: logfile })
    end

    def self.benchmark(&block)
      Benchmark.realtime do
        yield block
      end
    end

    def self.get_total_runtime(timings)
      timings.inject(0) do |sum, (_, v)|
        sum + v
      end
    end

    def self.post_response(url, cert_conf, request_body = nil, raise_on_failure: true)
      uri = URI.parse(url)
      https = build_auth(uri, cert_conf)

      request = Net::HTTP::Post.new(uri.request_uri)
      request['Content-Type'] = 'application/json'
      request.body = request_body if request_body

      res = https.request(request)
      if raise_on_failure
        case res
        when Net::HTTPSuccess then
          res
        else
          error(_('An error occurred posting to %{url}: HTTP %{http_code}, %{metadata}, %{body}.') % { url: url, http_code: res.code, metadata: res.to_hash.inspect, body: res.body })
        end
      else
        res
      end
    end

    def self.get_response(url, cert_conf, request_body = nil, raise_on_failure: true)
      uri = URI.parse(url)
      https = build_auth(uri, cert_conf)

      request = Net::HTTP::Get.new(uri.request_uri)
      request['Content-Type'] = 'application/json'
      request.body = request_body if request_body

      res = https.request(request)
      if raise_on_failure
        case res
        when Net::HTTPSuccess then
          res
        else
          error(_('An error occurred getting a response from %{url}: HTTP %{http_code}, %{metadata}, %{body}.') % { url: url, http_code: res.code, metadata: res.to_hash.inspect, body: res.body })
        end
      else
        res
      end
    end

    def self.extract_available_space_and_mount_point(dir)
      command = "df --block-size=1 --portability #{dir}"
      stdout, stderr, status = run_command(command, combine: false)
      unless status.exitstatus.zero?
        error(_('Could not calculate available space.'), command, stdout + "\n" + stderr)
      end
      last_line = stdout.split("\n").last
      free_space = last_line.split(' ')[3].to_i
      mount_point = last_line.split(' ')[5]
      return [free_space, mount_point]
    end

    def self.humanize_time(secs)
      if secs < 60
        _('%{seconds} sec') % { seconds: secs.round }
      else
        minutes, seconds = secs.divmod(60)
        if minutes < 60
          _('%{minutes} min, %{seconds} sec') % { minutes: minutes, seconds: seconds.round }
        else
          hours, minutes = minutes.divmod(60)
          _('%{hours} hr, %{minutes} min, %{seconds} sec') % { hours: hours, minutes: minutes, seconds: seconds.round }
        end
      end
    end

    def self.humanize_size(bytes)
      bytes = bytes.to_i
      units = %w[B KB MB GB TB].freeze

      if bytes < 1024
        exponent = 0
        num = bytes
      else
        max_exp  = units.size - 1

        exponent = (Math.log(bytes) / Math.log(1024)).to_i # convert to base
        exponent = max_exp if exponent > max_exp # we need this to avoid overflow for the highest unit
        num = (bytes / 1024.0**exponent).round(2)
      end
      "#{num} #{units[exponent]}"
    end

    # Helper class for printing the steps of a backup/restore
    class StepPrinter
      def initialize(total_steps)
        @step_counter = 1
        @total_steps = total_steps
      end

      def step(msg)
        Puppet.info(_('Step %{step_counter} of %{total_steps}: %{msg}') % { step_counter: @step_counter, total_steps: @total_steps, msg: msg })
        @step_counter += 1
      end
    end
  end
end
