# frozen_string_literal: true

require 'bolt/applicator'
require 'bolt/error'

module PlanRunner
  module BoltOverrides
    class Applicator < Bolt::Applicator
      def initialize(inventory, executor, config)
        @config = config

        # rubocop:disable Lint/UselessAssignment
        super(
          inventory,
          executor,
          # These positional arguments are used for ast
          # compilation except for plugin_dirs is used by
          # `Applicator#build_plugin_tarball`
          modulepath = nil,
          plugin_dirs = nil,
          project = nil,
          pdb_client = nil,
          hiera_config = nil,
          max_compiles = nil,
          apply_settings = nil
        )
        # rubocop:enable Lint/UselessAssignment

        # The parent class creates a Concurrent::ThreadPoolExecutor
        # pool for concurrent applies/ast complilations.
        @pool = nil
      end

      # args[0] = TargetSpec
      # args[1] = Hash with all apply options
      # apply_body = string with the Puppet code (not serialized)
      # scope = plan variables as JSON
      def apply(args, apply_body, scope)
        raise ArgumentError, 'apply requires a TargetSpec' if args.empty?
        raise ArgumentError, 'apply requires at least one statement in the apply block' if apply_body.nil?

        type0 = Puppet.lookup(:pal_script_compiler).type('TargetSpec')
        Puppet::Pal.assert_type(type0, args[0], 'apply targets')
        targets = @inventory.get_targets(args[0])

        options = {}
        if args.count > 1
          type1 = Puppet.lookup(:pal_script_compiler).type('Hash[String, Data]')
          Puppet::Pal.assert_type(type1, args[1], 'apply options')
          options = args[1].transform_keys { |k| k.sub(/^_/, '').to_sym }
        end

        plan_vars = scope.to_hash(true, true)
        %w[trusted server_facts facts].each { |k| plan_vars.delete(k) }

        plan_vars = Puppet::Pops::Serialization::ToDataConverter.convert(plan_vars,
                                                                         rich_data: true,
                                                                         symbol_as_string: true,
                                                                         type_by_reference: true,
                                                                         local_reference: true)

        ast = Puppet::Pops::Serialization::ToDataConverter.convert(apply_body,
                                                                   rich_data: true,
                                                                   symbol_to_string: true)
        # CODEREVIEW: This is our chance to interact with the resolved targetspec to build custom facts
        # We build a hash with target-name => trusted facts and pass that through to clojure in compile_context
        # Is this the right thing to do?
        trusted_facts = targets.each_with_object({}) do |target, acc|
          node = Puppet::Node.from_data_hash('name' => target.name,
                                             'parameters' => { 'clientcert' => target.name })
          trusted = Puppet::Context::TrustedInformation.local(node).to_h
          acc[target.name] = trusted
        end

        # In order to preserver order of plan vars serialization we pass them through ruby -> clojure
        # -> json etc as an array of single k,v pair hashes.
        ordered_plan_vars = plan_vars.map { |k, v| { k => v } }
        compile_context = { ast: ast, plan_vars: ordered_plan_vars, trusted_facts: trusted_facts }
        result = @executor.apply_puppet_code(targets, compile_context, options)

        raise Bolt::ApplyFailure, result if !result.ok && !options[:catch_errors]

        result
      end
    end
  end
end
