# frozen_string_literal: true

require 'bolt/inventory'

# The PlanRunner version of Inventory should mostly execute in the same way that
# the Bolt version does: with a few exceptions:
#
# 1. The config key for targets is always empty in PE, from the context of
#    ruby bolt Target objects are all references to an external inventory
#    that bolt doesn't have access to, so the configuration of each host
#    is unavailable in plans (at least for now).
#
# 2. Since the config key is always empty: in PE you cannot set any config
#    values for a target. This means you cannot:
#      * Use `set_config`
#      * Use a transport name as the scheme in a target URI
#      * Update any targets to run with a different transport mid-plan
#
# 3. Group creation/access is unavailable in PE. Since PE is not using a bolt
#    inventory we have no ability to create/read/write any groups. Thus
#    any attempts to use group actions in plans will fail.
#
# 4. The features array for targets is always empty in PE, from the context of
#    ruby bolt Target objects are all references to an external inventory
#    that bolt doesn't have access to, so features are also unavailable in the
#    PE context
#
# 5. Since the features array is always empty: in PE you cannot set any features
#    using `set_feature`
module PlanRunner
  module BoltOverrides
    class Inventory
      class Inventory < Bolt::Inventory::Inventory
        # Forbid setting config (connection info) on target in PE
        def set_config(_target, _key_or_key_path, _value)
          raise Bolt::Inventory::ValidationError.new(
            'Setting target configuration from a plan is not supported in Puppet Enterprise', nil
          )
        end

        # Forbid setting features (connection info) on target in PE
        def set_feature(_target, _feature, _value = true)
          raise Bolt::Inventory::ValidationError.new('Target features are not supported in Puppet Enterprise', nil)
        end

        # Override this method such that groups are forbidden
        def resolve_name(target)
          if group_lookup[target]
            # This should only match the `all` group FORBID in PE
            raise Bolt::Inventory::ValidationError.new('Groups are not supported in Puppet Enterprise', nil)
          else
            # Try to wildcard match targets in inventory
            # Ignore case because hostnames are generally case-insensitive
            regexp = Regexp.new("^#{Regexp.escape(target).gsub('\*', '.*?')}$", Regexp::IGNORECASE)

            targets = groups.all_targets.grep(regexp)
            targets += groups.target_aliases.select { |target_alias, _target| target_alias =~ regexp }.values

            if targets.empty?
              raise(WildcardError, target) if target.include?('*')

              [target]
            else
              targets
            end
          end
        end
        private :resolve_name

        # Override this method to forbid setting transport in uri component Bolt's
        # inventory includes an 'ext_glob' keyword argument here that is unused in
        # PE, so just ignore it.
        def expand_targets(targets, **_kwargs)
          case targets
          when Bolt::Target
            targets
          when Array
            targets.map { |tish| expand_targets(tish) }
          when String
            # Expand a comma-separated list
            targets.split(/[[:space:],]+/).reject(&:empty?).map do |name|
              ts = resolve_name(name)
              ts.map do |t|
                # If the target doesn't exist, evaluate it from the inventory. Then return a Bolt::Target.
                if t =~ %r{^[^:]+://}
                  # Given that `parse_uri` method has been migrated to the Bolt::Inventory::Target class
                  # we need to check here if a transport is set in the URI.
                  msg = "Invalid Target: '#{t}'. Using schemes in Target URIs is not available in Puppet Enterprise"
                  raise Bolt::ParseError, msg
                end
                @targets[t] = create_target_from_inventory(t) unless @targets.key?(t)
                Bolt::Target.new(t, self)
              end
            end
          end
        end
        private :expand_targets

        # Override this method so Target.new cannot configure connection information
        def create_target_from_hash(data)
          # Scrub config and features FORBID in PE
          cleaned_data = data.merge({ 'config' => {}, 'features' => [] })
          if data.include?('config')
            msg = "Cannot instantiate new target with 'config'. " \
                  "Setting target configuration from a plan is not supported in Puppet Enterprise"
            raise Bolt::Inventory::ValidationError.new(msg, nil)
          elsif data.include?('features')
            msg = "Cannot instantiate new target with 'features'. " \
                  "Setting target features from a plan is not supported in Puppet Enterprise"
            raise Bolt::Inventory::ValidationError.new(msg, nil)
          end
          # If target already exists, delete old and replace with new, otherwise add to new to all group
          new_target = Bolt::Inventory::Target.new(cleaned_data, self)
          existing_target = @targets.key?(new_target.name)

          validate_target_from_hash(new_target)
          @targets[new_target.name] = new_target

          if existing_target
            clear_alia_from_group(groups, new_target.name)
          else
            add_to_group([new_target], 'all')
          end

          groups.insert_alia(new_target.name, Array(new_target.target_alias)) if new_target.target_alias

          new_target
        end

        # Override to only allow adding to `all` group
        def add_to_group(targets, desired_group)
          unless desired_group == 'all'
            raise Bolt::Inventory::ValidationError.new('Groups are not supported in Puppet Enterprise', nil)
          end

          if group_names.include?(desired_group)
            targets.each do |target|
              if group_names.include?(target.name)
                raise Bolt::Inventory::ValidationError.new(
                  "Group #{target.name} conflicts with target of the same name", target.name
                )
              end

              # Add the inventory copy of the target
              add_target(groups, @targets[target.name], desired_group)
            end
          else
            raise Bolt::Inventory::ValidationError.new("Group #{desired_group} does not exist in inventory", nil)
          end
        end
      end
    end
  end
end
