# frozen_string_literal: true

require 'hocon'
require 'logging'
require 'plan_runner/service/encryption'
require 'plan_runner/service/http_client'
require 'pe_bolt_server/config'

module PlanRunner
  module Service
    class ConfigLoader
      def initialize(confdir = '/etc/puppetlabs/pe-plan-runner/conf.d')
        @logger = Logging.logger[self]
        @confdir = confdir
      end

      def defaults
        {
          codedir:       '/opt/puppetlabs/server/data/orchestration-services/code',
          log_level:     'debug',
          log_file:      '/var/log/puppetlabs/plan-runner/pe-plan-runner.log',
          max_plans:     100,
          default_pdb:   {},
          pdb_instances: {}
        }
      end

      def load_ca_cert(path)
        if File.exist?(path)
          File.read(path).scan(/-----BEGIN CERTIFICATE-----.*?-----END CERTIFICATE-----/m).map do |cert|
            OpenSSL::X509::Certificate.new(cert)
          end
        else
          @logger.error("Could not load CA certificate(s) at '#{path}'")
        end
      end

      def load_crl(path)
        if File.exist?(path)
          File.read(path).scan(/-----BEGIN X509 CRL-----.*?-----END X509 CRL-----/m).map do |crl|
            OpenSSL::X509::CRL.new(crl)
          end
        else
          @logger.error("Could not load CRL(s) at '#{path}'")
        end
      end

      def load_client_cert(path)
        if File.exist?(path)
          OpenSSL::X509::Certificate.new(File.read(path))
        else
          @logger.error("Could not load client certificate at '#{path}'")
        end
      end

      def load_private_key(path)
        if File.exist?(path)
          OpenSSL::PKey::RSA.new(File.read(path))
        else
          @logger.error("Could not load private key at '#{path}'")
        end
      end

      def load(overrides = {})
        raise "Could not find config directory: '#{@confdir}'" unless File.exist?(@confdir)

        parsed_config = Dir[@confdir + '/*.conf'].reduce({}) do |conf, file|
          parsed = Hocon.load(file)
          conf.merge(parsed)
        end

        pkey                   = parsed_config.dig('pe-plan-runner', 'certs', 'ssl-key')
        cert                   = parsed_config.dig('pe-plan-runner', 'certs', 'ssl-cert')
        cacert                 = parsed_config.dig('pe-plan-runner', 'certs', 'ssl-ca-cert')
        crl                    = parsed_config.dig('pe-plan-runner', 'certs', 'ssl-crl-path')
        orch_url               = parsed_config.dig('pe-plan-runner', 'orchestrator-url')
        log_level              = parsed_config.dig('pe-plan-runner', 'log-level')
        log_file               = parsed_config.dig('pe-plan-runner', 'log-file')
        max_plans              = parsed_config.dig('pe-plan-runner', 'max-plans')
        codedir                = parsed_config.dig('pe-plan-runner', 'codedir')
        default_pdb            = parsed_config.dig('pe-plan-runner', 'puppetdb')
        pdb_instances          = parsed_config.dig('pe-plan-runner', 'puppetdb-instances')
        license_api            = parsed_config.dig('license-check', 'license-api')
        license_check_interval = parsed_config.dig('license-check', 'check-interval-seconds')
        license_check_retries  = parsed_config.dig('license-check', 'retries')

        pdb_instances ||= {}
        if default_pdb.nil? && pdb_instances.is_a?(Hash)
          default_pdb = pdb_instances.values.first
        end

        config = defaults.merge(
          {
            private_key: pkey,
            ca_cert: cacert,
            cert: cert,
            crl: crl,
            orch_url: orch_url,
            log_level: log_level,
            log_file: log_file,
            codedir: codedir,
            max_plans: max_plans,
            default_pdb: default_pdb,
            pdb_instances: pdb_instances,
            license_api: license_api,
            license_check_interval: license_check_interval,
            license_check_retries: license_check_retries,
            # Entitlements should never be set by config on disk and should not
            # be configurable by the user. The only way to set any entitlements
            # should be to run `load_entitlements`
            entitlements: PEBoltServer::EntitlementsData.new
          }.compact,
          overrides
        )

        # Once we know where the log file is, immediately reset the root
        # logger with an appender for that file. After this line we will
        # be printing logs to the logfile, and no longer stdout
        PlanRunner::LoggingHelper.reinit_root_logger(config[:log_file], config[:log_level])
        # The first thing to do once we have a real logger to a file is
        # indicate to the user that something is happening
        @logger.info("Configuration loaded")

        config[:encryption_key] = ::PlanRunner::Service::RSAEncryptionKey.new
        config[:private_key]    = load_private_key(config[:private_key])
        config[:cert]           = load_client_cert(config[:cert])
        config[:ca_cert]        = load_ca_cert(config[:ca_cert])
        config[:crl]            = load_crl(config[:crl])

        %i[private_key ca_cert cert crl orch_url log_level codedir encryption_key max_plans].each do |key|
          raise "Missing required configuration: #{key}" unless config[key]
        end

        load_entitlements(config)
        config
      end

      def load_entitlements(config)
        client = HttpClient.new("", config)
        config[:entitlements].load_logger(@logger)
        # It would be nice if both pe-bolt-server and pe-plan-runner used similar
        # config loading schemes, especially since bolt-server's seems unnecessarily
        # complicated. However for now that work is left to a later date.
        #
        # Instead format a config hash like the one bolt-server's EntitlementsData class
        # expects
        entitlements_check_config = { 'license-check' => {
          'license-api' => config[:license_api],
          'retries' => config[:license_check_retries],
          'check-interval-seconds' => config[:license_check_interval]
        } }
        config[:entitlements].start_entitlements_check(client, entitlements_check_config)
      rescue StandardError => e
        @logger.debug("Starting entitlements check failed with #{e.message}")
      end
    end
  end
end
