# frozen_string_literal: true

require 'openssl'
require 'base64'
require 'json'
require 'bolt/error'

module PlanRunner
  module Service
    class RSAEncryptionKey
      def initialize(key: nil)
        @logger = Logging.logger[self]
        @logger.info("Creating new RSA key")
        @key = OpenSSL::PKey::RSA.new(key || 4096)
      end

      def public_key
        @key.public_to_pem
      end

      def private_key
        @key.private_to_pem
      end

      def decrypt(raw_data)
        @key.decrypt(Base64.decode64(raw_data), rsa_padding_mode: "oaep", rsa_oaep_md: "sha256")
      end
    end

    class AESEncryptionCipher
      def initialize(key)
        @logger = Logging.logger[self]
        @cipher_instance = OpenSSL::Cipher.new('aes-256-gcm')
        @cipher_key = key
      end

      def decrypt(raw_data)
        auth_tag_len, iv, data = if raw_data.is_a? String
                                   JSON.parse(raw_data)
                                 else
                                   raw_data
                                 end.map { |data| Base64.decode64(data) }

        # Pull out the auth tag from the encrypted data sent by the encrypting method. Java's
        # encryption methods put the auth tag at the end of the byte array that comprises the
        # encrypted data itself. The thing we are provided over the wire is the length (in bits)
        # of that tag that was put at the end of the encrypted data.
        #
        # So pull out that tag from the end and then strip those bytes from the end of the data
        auth_tag_len = auth_tag_len.to_i / 8
        auth_tag = data.byteslice(data.bytesize - auth_tag_len, auth_tag_len)
        data = data.byteslice(0, data.bytesize - auth_tag_len)

        # Once we have the auth tag we can now properly set up the cipher:
        #
        # * Make sure the cipher is set up for decryption
        # * The cipher is reset between each encryption, so set the cipher key again
        # * The Initialization vector was sent directly over the wire, set that
        # * The auth tag has been gathered, set that
        # * There's no auth_data to be added to the cipher (we don't use this), but OpenSSL
        #   requires that to be set even if it's empty.
        @cipher_instance.decrypt
        @cipher_instance.key = @cipher_key
        @cipher_instance.iv = iv
        @cipher_instance.auth_tag = auth_tag
        @cipher_instance.auth_data = ""

        # Now we can actually decrypt the thing.
        @cipher_instance.update(data) + @cipher_instance.final
      rescue StandardError
        # Swallow any actual decryption error, since the message may contain
        # sensitive data. Raise a generic bolt error
        msg = "Failed to decypt sensitive data"
        @logger.error(msg)
        raise Bolt::Error.new(msg, 'bolt.plan-executor/decryption-failed')
      ensure
        # Reset the cipher instance so we can use it again
        @cipher_instance.reset
      end
    end
  end
end
