# frozen_string_literal: true

require 'pathname'
require 'bolt/error'

# Downloads the given file or directory from the given set of targets and saves it to a directory
# matching the target's name under the given destination directory. Returns the result from each
# download. This does nothing if the list of targets is empty.
#
# > **Note:** Existing content in the destination directory is deleted before downloading from
# > the targets.
#
# > **Note:** Not available in apply block
Puppet::Functions.create_function(:download_file, Puppet::Functions::InternalFunction) do
  # Download a file or directory.
  # @param source The absolute path to the file or directory on the target(s).
  # @param destination The relative path to the destination directory on the local system. Expands
  #                    relative to `<project>/downloads/`.
  # @param targets A pattern identifying zero or more targets. See {get_targets} for accepted patterns.
  # @param options A hash of additional options.
  # @option options [Boolean] _catch_errors Whether to catch raised errors.
  # @option options [String] _run_as User to run as using privilege escalation.
  # @return A list of results, one entry per target, with the path to the downloaded file under the
  #         `path` key.
  # @example Download a file from multiple Linux targets to a destination directory
  #   download_file('/etc/ssh/ssh_config', '~/Downloads', $targets)
  # @example Download a directory from multiple Linux targets to a project downloads directory
  #   download_file('/etc/ssh', 'ssh', $targets)
  # @example Download a file from multiple Linux targets and compare its contents to a local file
  #   $results = download_file($source, $destination, $targets)
  #
  #   $local_content = file::read($source)
  #
  #   $mismatched_files = $results.filter |$result| {
  #     $remote_content = file::read($result['path'])
  #     $remote_content == $local_content
  #   }
  dispatch :download_file do
    param 'String[1]', :source
    param 'String[1]', :destination
    param 'Boltlib::TargetSpec', :targets
    optional_param 'Hash[String[1], Any]', :options
    return_type 'ResultSet'
  end

  # Download a file or directory, logging the provided description.
  # @param source The absolute path to the file or directory on the target(s).
  # @param destination The relative path to the destination directory on the local system. Expands
  #                    relative to `<project>/downloads/`.
  # @param targets A pattern identifying zero or more targets. See {get_targets} for accepted patterns.
  # @param description A description to be output when calling this function.
  # @param options A hash of additional options.
  # @option options [Boolean] _catch_errors Whether to catch raised errors.
  # @option options [String] _run_as User to run as using privilege escalation.
  # @return A list of results, one entry per target, with the path to the downloaded file under the
  #         `path` key.
  # @example Download a file from multiple Linux targets to a destination directory
  #   download_file('/etc/ssh/ssh_config', '~/Downloads', $targets, 'Downloading remote SSH config')
  dispatch :download_file_with_description do
    param 'String[1]', :source
    param 'String[1]', :destination
    param 'Boltlib::TargetSpec', :targets
    param 'String', :description
    optional_param 'Hash[String[1], Any]', :options
    return_type 'ResultSet'
  end

  def download_file(source, destination, targets, options = {})
    download_file_with_description(source, destination, targets, nil, options)
  end

  def download_file_with_description(source, destination, targets, description = nil, options = {})
    unless Puppet[:tasks]
      raise Puppet::ParseErrorWithIssue
        .from_issue_and_stack(Bolt::PAL::Issues::PLAN_OPERATION_NOT_SUPPORTED_WHEN_COMPILING, action: 'download_file')
    end
    # TODO: https://tickets.puppetlabs.com/browse/PE-29861
    raise Puppet::Error, "download_file is currently not supported in PE plans"
  end
end