# frozen_string_literal: true

require 'minitar'
require 'open3'
require 'pe_installer/logger'
require 'resolv'
require 'zlib'

module PeInstaller

  # Catch all module for utility methods we might want to use from multiple classes.
  module Util

    extend PeInstaller::Logger

    # Given a valid tarball, return the contents of a matching file name.
    # It will unwrap a tgz or tar.gz zipped tarball based on extension.
    #
    # Note: this is just intended for simple lookups of metadata text files
    # from PE tarballs.
    #
    # @param file [String] name/relative path of the file within the tarball.
    #   Ex: packages/bootstrap-metadata
    # @param tarball_path [String] absolute path to the tarball.
    # @return [String] contents of the file.
    # @raise [PeInstaller::Error] if the file does not exist, is not a tar,
    #   or the file isn't found within the tar.
    def self.get_file_from_tar(file, tarball_path)
      extractor = lambda do |file_name, io|
        contents = nil
        begin
          Minitar::Input.each_entry(io) do |e|
            if e.file? && e.name.end_with?("/#{file_name}")
              contents = e.read
              break
            end
          end
        rescue ArgumentError => e
          raise(
            PeInstaller::Error,
            "Unable to find #{file}; '#{tarball_path}' is not a valid tarball. Original minitar error: '#{e}'"
          )
        end
        contents
      end

      info("Looking up '#{file}' in '#{tarball_path}'")

      if !File.exist?(tarball_path)
        raise(PeInstaller::Error, "Unable to get #{file}; '#{tarball_path}' does not exist.")
      end

      fileio = File.new(tarball_path, 'r')
      file_contents =
        case File.extname(tarball_path)
        when '.tgz', '.gz'
          z = Zlib::GzipReader.new(fileio)
          extractor.call(file, z)
        else
          # assume '.tar'
          extractor.call(file, fileio)
        end

      raise(PeInstaller::Error, "Failed to find '#{file}' in '#{tarball_path}'.") if file_contents.nil?

      file_contents
    end

    # Given either an packed PE tarball, or the tarball, lookup and return
    # packages/bootstrap-metadata as a Hash.
    #
    # @param tarball_dir [String] absolute path to a PE tarball directory.
    # @param tarball [String] absolute path to a PE tarball.
    # @return [Hash]
    # @raise [PeInstaller::Error] if unable to find or load boostrap-metadata.
    def self.load_bootstrap_metadata_from(tarball_dir: nil, tarball: nil)
      if !tarball_dir.nil?
        info("Looking up bootstrap-metadata in tarball_dir: '#{tarball_dir}'.")
        metadata_file = "#{tarball_dir}/packages/bootstrap-metadata"
        raise(PeInstaller::Error, "Unable to find #{metadata_file}.") if !File.exist?(metadata_file)
        contents = File.read(metadata_file)
      elsif !tarball.nil?
        contents = PeInstaller::Util.get_file_from_tar('packages/bootstrap-metadata', tarball)
      else
        raise(
          PeInstaller::Error,
          'Unable to load bootstrap-metadata. Neither a directory nor a tarball path were given.'
        )
      end

      metadata = contents.split("\n").map { |l| l.split('=') }.to_h
      metadata.transform_keys(&:downcase)
    end

    # @return [Array<String>] array of all ip addresses from the local network
    #   interfaces returned by Facter.
    def self.get_local_ip_addresses
      require 'facter'
      # Recent facter seems to have removed the option to set these,
      # so we're digging down into the OptionStore to set them instead.
      # Facter::Options[:custom_facts] = false
      # Facter::Options[:external_facts] = false
      Facter::OptionStore.no_custom_facts = true
      Facter::OptionStore.no_external_facts = true
      interfaces = Facter.fact('networking.interfaces').value || {}
      # At the moment, this ignores ipv6-only interfaces
      interfaces.values.map do |i|
        i['bindings'].map { |b| b['address'] } if i['bindings']
      end.flatten
    end

    # Set transport to local if the Target's hostname resolves as a local ip address.
    #
    # Since it's the Bolt::Inventory::Target that is updated, future target lookups
    # from the same Inventory instance should carry any changes made.
    #
    # @param target [Bolt::Target] the target to check.
    # @return [Bolt::Target] with transport updated to 'local' if warranted.
    def self.resolve_local_target(target)
      return if target.nil?

      uri_string = target.uri || target.name
      info("Resolving #{uri_string} to see if it corresponds to a local address.")
      local_addresses = PeInstaller::Util.get_local_ip_addresses
      debug("Local ip addresses: #{local_addresses}")
      uri = Bolt::Inventory::Target.parse_uri(uri_string)
      hostname = uri.host
      debug("Target uri: #{uri}, hostname: #{hostname}")
      host_addresses = Resolv.getaddresses(hostname)
      debug("Host addresses: #{host_addresses}")
      if !(local_addresses & host_addresses).empty?
        target.inventory_target.set_config('transport', 'local')
        debug("Reset #{hostname} target transport to local.")
      end
      target
    rescue Resolv::ResolvError, Resolv::ResolvTimeout => e
      warn("Unable to resolv #{hostname}: #{e}")
      target
    end
  end
end
