#!/usr/bin/env ruby
# frozen_string_literal: true

require 'json'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

# Lock a NETCONF datastore
#
# This task locks a datastore to prevent other sessions from modifying it.
# Remember to unlock it when done or the lock will persist until session closes.

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Smart default: use candidate if supported, otherwise running
  datastore = session.task_params['datastore']
  if datastore.nil?
    datastore = session.supports_candidate? ? 'candidate' : 'running'
  end

  begin
    session.lock(datastore)

    session.report_result({
                            status: 'success',
      message: "Successfully locked #{datastore} datastore",
      datastore: datastore,
      session_id: session.connection.session_id,
      warning: 'Remember to unlock the datastore when done with your changes'
                          })
  rescue PuppetX::Puppetlabs::Netconf::NetconfError::LockDenied => e
    # Parse who holds the lock
    lock_holder = e[:session_id] || e[:'session-id'] || e[:info_session_id] || e[:'info_session-id']

    session.report_result({
                            status: 'error',
      message: "Cannot lock #{datastore}: already locked",
      datastore: datastore,
      lock_holder: (lock_holder == '0') ? 'non-NETCONF entity (CLI/web/uncommitted changes)' : "session #{lock_holder}",
      recommendation: if lock_holder == '0'
                        "Use 'bolt task run netconf_discard_changes' to clear uncommitted changes"
                      else
                        "Use 'bolt task run netconf_recover_lock' to recover the lock"
                      end
                          })
  rescue => e
    session.report_result({
                            status: 'error',
      message: "Failed to lock #{datastore}: #{e.message}",
      datastore: datastore
                          })
  end
end

puts result.to_json
