# Generates the internal certificates that were previously created by the installer.
#
# See https://tickets.puppet.com/browse/PE-13583 for a discussion of their fates.
class pe_install::prepare::certificates {
  include pe_install::params

  # Having trouble making path work with exec and puppet, won't recognize the path
  $puppetserver_cmd = "${pe_install::params::puppetlabs_bin_dir}/puppetserver"
  $puppet_cmd = "${pe_install::params::puppet_bin_dir}/puppet"

  if $::pe_install::is_ca {
    if (pe_empty($::pe_install::puppet_master_dnsaltnames)) {
      $_dns_alt_name_flag = ''
    } else {
      $_dns_cert_names_formated = pe_join($::pe_install::puppet_master_dnsaltnames, ',')
      $_dns_alt_name_flag = " --subject-alt-names '${_dns_cert_names_formated}'"
    }

    $_ca_name = "Puppet Enterprise CA generated at ${pe_formatted_date('+%Y-%m-%d %H:%M:%S %z')}"
    $_ca_cert_exec_notify_defaults = [
      Service['pe-puppetserver'],
      Service['pxp-agent'],
    ]

    if pe_install::puppet_service_managed == true {
      $_ca_cert_exec_notify = $_ca_cert_exec_notify_defaults + Service['puppet']
    } else {
      $_ca_cert_exec_notify = $_ca_cert_exec_notify_defaults
    }

    # Ensure that the CA certificate (chain) is imported or generated
    $ca_cert = "${pe_install::params::ca_dir}/ca_crt.pem"
    if $::pe_install::signing_ca != undef {
      $_bundle = $::pe_install::signing_ca['bundle']
      $_crl_chain = $::pe_install::signing_ca['crl_chain']
      $_private_key = $::pe_install::signing_ca['private_key']
      $_intermediate_ca_flags = "--cert-bundle ${_bundle} --crl-chain ${_crl_chain} --private-key ${_private_key}"

      exec { 'import provided signing ca':
        command => "${puppetserver_cmd} ca import ${_intermediate_ca_flags}${_dns_alt_name_flag}",
        creates => $ca_cert,
        notify  => $_ca_cert_exec_notify,
        before  => [Exec['generate node cert'], Puppet_enterprise::Set_owner_group_permissions['/etc/puppetlabs/puppet/ssl/certs/ca.pem']],
        require => Package['pe-puppetserver'],
      }
    } else {
      exec { 'generate ca cert':
        command => "${puppetserver_cmd} ca setup --certname ${::clientcert} --ca-name '${_ca_name}'${_dns_alt_name_flag}",
        creates => $ca_cert,
        notify  => $_ca_cert_exec_notify,
        before  => Exec['generate node cert'],
        require => Package['pe-puppetserver'],
      }
    }

    # Ensure that we generate a certificate for the ca node for Puppet.
    # Generally this is created as a side effect of the ca setup/import.
    # But if the primary node cert has been deleted, or a restore is happening
    # on a new node, we may need to regenerate it.
    $signed_ca_node_cert = "${pe_install::params::ssl_dir}/certs/${::clientcert}.pem"
    $cert_content = file($signed_ca_node_cert, '/dev/null')
    $is_empty_cert = empty($cert_content)
    if $is_empty_cert {
      $puppetserver_refs = [
        Package['pe-puppetserver'],
        Pe_install::Stop_service['stop pe-puppetserver before cert generation'],
      ]

      pe_install::stop_service { 'stop pe-puppetserver before cert generation':
        service          => 'pe-puppetserver',
        before_reference => [],
      }
    } else {
      $puppetserver_refs = [ Package['pe-puppetserver'], ]
    }

    exec { 'generate node cert':
      command   => "${puppetserver_cmd} ca generate --certname ${::clientcert} --ca-client${_dns_alt_name_flag}",
      creates   => $signed_ca_node_cert,
      notify    => $_ca_cert_exec_notify,
      require   => $puppetserver_refs,
    }

    file { '/etc/puppetlabs/puppet/autosign.conf':
      ensure => file,
    }

    puppet_enterprise::set_owner_group_permissions { "/etc/puppetlabs/puppet/ssl/certs/ca.pem":
      file_mode   => '0644',
      owner       => 'pe-puppet',
      group       => 'pe-puppet',
      target_type => 'file',
    }

    $services_certnames = pe_unique(pe_flatten([
      $::pe_install::master_certname,
      $::pe_install::puppetdb_certname,
      $::pe_install::console_certname,
      $::pe_install::database_certname,
      $::pe_install::puppetdb_database_host,
      $::pe_install::console_database_host,
      $::pe_install::orchestrator_database_host
    ])).each |String $cert| {
      if $facts['clientcert'] != $cert {
        pe_file_line { "autosign ${cert}":
          ensure  => present,
          path    => '/etc/puppetlabs/puppet/autosign.conf',
          line    => $cert,
          require => File['/etc/puppetlabs/puppet/autosign.conf'],
        }
      }
    }
  }
  else {
    $_ca_find_exec_notify_defaults = [
      Service['pxp-agent']
    ]

    if pe_install::puppet_service_managed == true {
      $_ca_find_exec_notify = $_ca_find_exec_notify_defaults + Service['puppet']
    } else {
      $_ca_find_exec_notify = $_ca_find_exec_notify_defaults
    }

    # We can do this because any node we have submitted a csr for was added to
    # the autosign.conf on the primary (see above)
    exec { "retrieve cert for ${::clientcert}":
      command => "${puppet_cmd} ssl bootstrap --certname ${::clientcert} --ca_server ${::pe_install::ca_certname}",
      creates => "/etc/puppetlabs/puppet/ssl/certs/${::clientcert}.pem",
      notify  => $_ca_find_exec_notify
    }
  }
}
