# This class is for setting up the PuppetDB configuration file
#
# @param database_host [String] The hostname of the database that PuppetDB will
#        be running on
# @param confdir [String] The path to PuppetDB's confdir
# @param database_name [String] The name of the PuppetDB Database
# @param database_password [String] The password of the user
# @param database_port [Integer] The port that the database is running on
# @param database_properties [String] A url encoded string of JDBC options. This will replace the
#        default database property string which enables SSL connections.
# @param database_user [String] The user logging into the database
# @param gc_interval The interval, in minutes, at which garbage collection should occur
# @param gc_interval_expire_nodes The interval, in minutes, at which inactive nodes should be expired
# @param gc_interval_purge_nodes The interval, in minutes, at which nodes should be purged
# @param gc_interval_purge_reports The interval, in minutes, at which reports older than the ttl should be purged
# @param gc_interval_packages The interval, in minutes, at which packages no longer associated with nodes should be purged
# @param gc_interval_catalogs The interval, in minutes, at which catalog data no longer associated with nodes should be purged
# @param gc_interval_fact_paths The interval, in minutes, at which fact paths no longer associated with any factsets should be purged
# @param node_purge_ttl [String] The amount of time that must elapse before a
#        deactivated node is purged from PuppetDB
# @param node_ttl [String] The amount of time that must elapse before a node is
#        deactivated from PuppetDB
# @param report_ttl [String] The amount of time that must elapse before a report is deleted
# @param resource_events_ttl [Puppet_enterprise::Puppetdb_ttl] The amount of time that must elapse
#        before a resource event is deleted
# @param write_maximum_pool_size [Integer] this value will determine the maximum number
#        of actual connections to the database backend for write connections
class puppet_enterprise::puppetdb::database_ini(
  $database_host,
  $database_port,
  Integer $write_maximum_pool_size,
  $confdir                    = $puppet_enterprise::params::puppetdb_confdir,
  $database_name              = $puppet_enterprise::params::puppetdb_database_name,
  Optional[String[1]] $database_password = undef,
  Optional[String[1]] $migrator_password = undef,
  $database_properties        = '',
  $database_user              = $puppet_enterprise::params::puppetdb_database_user,
  $migrator_user              = $puppet_enterprise::params::puppetdb_migrator_user,
  Variant[Pattern[/^[0-9]+$/],Integer] $gc_interval = $puppet_enterprise::params::puppetdb_gc_interval,
  Optional[Integer] $gc_interval_expire_nodes       = undef,
  Optional[Integer] $gc_interval_purge_nodes        = undef,
  Optional[Integer] $gc_interval_purge_reports      = undef,
  Optional[Integer] $gc_interval_packages           = undef,
  Optional[Integer] $gc_interval_catalogs           = undef,
  Optional[Integer] $gc_interval_fact_paths         = undef,
  Optional[String] $node_purge_ttl = undef,
  $node_ttl                   = $puppet_enterprise::params::puppetdb_node_ttl,
  $report_ttl                 = $puppet_enterprise::params::puppetdb_report_ttl,
  Optional[Puppet_enterprise::Puppetdb_ttl] $resource_events_ttl = undef,
  Optional[Array[String]] $facts_blacklist = undef,
  Array[String] $facts_blocklist = pe_pick($facts_blacklist, []),
  Optional[Enum['literal','regex']] $facts_blacklist_type = undef,
  Enum['literal','regex'] $facts_blocklist_type = pe_pick($facts_blacklist_type, 'literal'),
  $migrate                    = $puppet_enterprise::params::puppetdb_migrate,
) inherits puppet_enterprise::params {
  include puppet_enterprise::packages
  include puppet_enterprise::puppetdb::database_ini::deprecations

  $section = 'database'
  $config_file = "${confdir}/${section}.ini"

  puppet_enterprise::puppetdb::shared_database_settings { $section :
    confdir             => $confdir,
    database_host       => $database_host,
    database_name       => $database_name,
    database_password   => $database_password,
    database_port       => Integer($database_port),
    database_user       => $database_user,
    database_properties => $database_properties,
    maximum_pool_size   => $write_maximum_pool_size,
  }

  #Set the defaults
  Pe_ini_setting {
    path    => $config_file,
    ensure  => present,
    section => $section,
    require => File[$config_file]
  }

  #Write Database Only Settings
  pe_ini_setting {'puppetdb_gc_interval':
    setting => 'gc-interval',
    value   => $gc_interval,
  }

  $optional_gc_settings = {
    'gc-interval-expire-nodes'  => $gc_interval_expire_nodes,
    'gc-interval-purge-nodes'   => $gc_interval_purge_nodes,
    'gc-interval-purge-reports' => $gc_interval_purge_reports,
    'gc-interval-packages'      => $gc_interval_packages,
    'gc-interval-catalogs'      => $gc_interval_catalogs,
    'gc-interval-fact-paths'    => $gc_interval_fact_paths,
  }
  $optional_gc_settings.each |$setting, $value| {
    $gc_ensure = $value ? {
      undef => 'absent',
      default => 'present',
    }

    pe_ini_setting { "puppetdb_${setting}":
      setting => $setting,
      value => $value,
      ensure => $gc_ensure,
    }
  }

  pe_ini_setting {'puppetdb_node_ttl':
    setting => 'node-ttl',
    value   => $node_ttl,
  }

  $node_purge_ttl_ensure = pe_empty($node_purge_ttl) ? {
    false => 'present',
    true  => 'absent',
  }

  pe_ini_setting {'puppetdb_node_purge_ttl':
    ensure  => $node_purge_ttl_ensure,
    setting => 'node-purge-ttl',
    value   => $node_purge_ttl,
  }

  pe_ini_setting {'puppetdb_report_ttl':
    setting => 'report-ttl',
    value   => $report_ttl,
  }

  $resource_events_ttl_ensure = pe_empty($resource_events_ttl) ? {
    false => 'present',
    true  => 'absent',
  }

  pe_ini_setting {'puppetdb_resource_events_ttl':
    ensure  => $resource_events_ttl_ensure,
    setting => 'resource-events-ttl',
    value   => $resource_events_ttl,
  }

  $joined_facts_blocklist = pe_join($facts_blocklist, ',')

  # delete blacklist setting to avoid conflict in PDB with blocklist
  pe_ini_setting {'puppetdb_facts_blacklist':
    setting => 'facts-blacklist',
    ensure  => absent,
  }

  # delete blacklist setting to avoid conflict in PDB with blocklist
  pe_ini_setting {'puppetdb_facts_blacklist_type':
    setting => 'facts-blacklist-type',
    ensure  => absent,
  }

  pe_ini_setting {'puppetdb_facts_blocklist':
    setting => 'facts-blocklist',
    value   => $joined_facts_blocklist,
  }

  pe_ini_setting {'puppetdb_facts_blocklist_type':
    setting => 'facts-blocklist-type',
    value   => $facts_blocklist_type,
  }

  pe_ini_setting {'puppetdb_migrator_username':
    setting => 'migrator-username',
    value   => $migrator_user,
  }

  pe_ini_setting {'puppetdb_migrate':
    setting => 'migrate',
    value   => $migrate,
  }

  if !pe_empty($migrator_password)  {
    pe_ini_setting {"puppetdb_migrator_password":
      setting => 'migrator-password',
      value   => $migrator_password,
    }
  }
}
