require 'json'
require 'fileutils'
require 'tmpdir'
require 'pe_backup_tools/utils'
require 'pe_backup_tools/utils/analytics_client'
require 'pe_backup_tools/utils/mixins'
require 'puppet'
require 'digest'

module PeBackupTools
  # This module provides functionality for the create command
  # create: Create a backup
  module Backup
    extend PeBackupTools::Utils::Mixins

    def self.gpg_check(gpgkey)
      gpg_cmd = "echo 'test' | gpg --trust-model always --recipient #{gpgkey} --encrypt"
      output, status = run_command(gpg_cmd)
      unless status.exitstatus.zero?
        error(_('GPG key check failed. Make sure GPG is installed and your GPG key is imported on this node to use the encryption option.'), gpg_cmd, output)
      end
    end

    def self.encrypt_archive(backup_dir, archive, gpgkey)
      backup_path = File.join(backup_dir, archive)
      encrypted_output = backup_path + '.gpg'
      gpg_cmd = "gpg --trust-model always --recipient #{gpgkey} --output #{encrypted_output} --encrypt #{backup_path}"
      output, status = run_command(gpg_cmd)
      if status.exitstatus.zero? && File.exist?(encrypted_output)
        size = File.size(encrypted_output)
        FileUtils.remove_entry(backup_path) if File.exist?(backup_path)
        return [encrypted_output, size]
      else
        error(_('GPG encryption of the archive failed. Unencrypted archive is available at %{backup_path}') % { backup_path: backup_path }, gpg_cmd, output)
      end
    end

    def self.gzip_local_backup(dir, backup_dir, archive)
      backup_path = File.join(backup_dir, archive)
      command = "gzip -c #{dir}/#{archive} > #{backup_path}"
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Failed to compress archive with gzip.'), command, output)
      end

      archive_sha = Digest::SHA256.file(backup_path).hexdigest
      archive_size = File.size(backup_path)
      return [archive_sha, archive_size]
    end

    def self.puppet_infrastructure_recover(environment, options)
      loop do
        debugflag = options[:debug] ? ' --debug' : ''
        command = "/opt/puppetlabs/bin/puppet-infrastructure recover_configuration --pe-environment #{environment}#{debugflag}"
        output, status = run_command(command)
        break if status.exitstatus.zero?
        if status.exitstatus == 17
          Puppet.info(_('Puppet agent run currently in progress. Waiting 30 seconds before trying again.'))
          sleep(30)
        else
          error(_('Command `puppet infrastructure recover_configuration` failed.'), command, output)
        end
      end
    end

    def self.primary?(options = {})
      cert_conf = PeBackupTools::Utils.get_cert_conf(Puppet[:certname])
      puppetdb_port = options[:puppetdb_port] || 8081
      begin
        data = PeBackupTools::Utils::PuppetDB.get_class_for_certname('puppet_enterprise', Puppet[:certname], Puppet[:certname], cert_conf, puppetdb_port)
        if data.nil?
          warning(_('Could not fetch the puppet_enterprise class for this node from PuppetDB. Are you sure you are running on the primary?'))
          return false
        end
        data.dig('parameters', 'puppet_master_host') == Puppet[:certname]
      rescue PeBackupTools::Utils::PuppetDB::PuppetDBQueryError => e
        # We don't want to block people from backing up if we're unable to do a PDB query to find this info
        warning(_("Error attempting to fetch the puppet_enterprise class for this node from PuppetDB to verify this node is the primary. We'll assume you're running this command on the primary. Error: %{message}" % { message: e.message }))
        true
      end
    end

    def self.monolithic?(options = {})
      cert_conf = PeBackupTools::Utils.get_cert_conf(Puppet[:certname])
      puppetdb_port = options[:puppetdb_port] || 8081
      begin
        data = PeBackupTools::Utils::PuppetDB.get_class_for_certname('puppet_enterprise', Puppet[:certname], Puppet[:certname], cert_conf, puppetdb_port)
        if data.nil? # Probably means the node doesn't have the puppet_enterprise class applied
          warning(_('Could not fetch the puppet_enterprise class for this node from PuppetDB. Are you sure you are running on the primary?'))
          return false
        end
        puppet_master_host = data.dig('parameters', 'puppet_master_host')
        console_host = data.dig('parameters', 'console_host')
        database_host = data.dig('parameters', 'database_host')
        puppetdb_host = data.dig('parameters', 'puppetdb_host')
        puppet_master_host == console_host && puppet_master_host == database_host && puppet_master_host == puppetdb_host
      rescue PeBackupTools::Utils::PuppetDB::PuppetDBQueryError => e
        # We don't want to block people from backing up if we're unable to do a PDB query to find this info
        warning(_("Error attempting to fetch the puppet_enterprise class for this node from PuppetDB to verify this is not a split architecture. We'll assume your architecture is not a split architecture. Error: %{message}" % { message: e.message }))
        true
      end
    end

    def self.validate_backup_location(backup_dir, backup_filename)
      error(_('Specified backup directory %{dir} does not exist.') % { dir: backup_dir }) unless Dir.exist?(backup_dir)
      error(_('Specified backup directory %{dir} is not writable.') % { dir: backup_dir }) unless File.writable?(backup_dir)
      error(_('Specified backup name %{filename} cannot be a file path.') % { filename: backup_filename }) if backup_filename != File.basename(backup_filename)
      backup_location = File.join(backup_dir, backup_filename)
      error(_('File already exists at desired location %{filepath}.') % { filepath: backup_location }) if File.exist?(backup_location)
    end

    def self.validate_postgres_writable(dir)
      command = "su - pe-postgres -s /bin/bash -c 'if [ -w #{dir} ] ; then exit 0; else exit 1; fi'"
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Staging directory %{dir} inside the backup directory is not writable by pe-postgres user.') % { dir: dir }, command, output)
      end
    end

    def self.write_metadata_file(dir, archive, backup_metadata)
      File.write(File.join(dir, 'backup_metadata.json'), JSON.pretty_generate(backup_metadata))
      command = "tar -P -C '#{dir}' -rf #{dir}/#{archive} backup_metadata.json"
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Failed to add metadata file to backup.'), command, output)
      end
    end

    def self.global_excluded_paths
      %w[
        /opt/puppetlabs/puppet
        /opt/puppetlabs/installer
        /opt/puppetlabs/server/pe_build
        /opt/puppetlabs/server/data/packages
        /opt/puppetlabs/server/apps
        /opt/puppetlabs/server/data/analytics/analytics
        /opt/puppetlabs/server/data/postgresql
        /opt/puppetlabs/server/data/puppetserver/yaml/facts
        /opt/puppetlabs/server/data/code-manager/worker-caches
        /opt/puppetlabs/server/data/environments/enterprise/modules
        /opt/puppetlabs/server/data/puppetserver/reports
        /opt/puppetlabs/server/data/puppetserver/server_data
        /opt/puppetlabs/server/data/puppetserver/vendored-jruby-gems
        /opt/puppetlabs/bin
        /opt/puppetlabs/client-tools
        /opt/puppetlabs/puppet-metrics-collector
        /opt/puppetlabs/server/share
        /etc/puppetlabs/nginx/conf.d/http_redirect.conf
        /etc/puppetlabs/nginx/conf.d/proxy.conf
        /etc/puppetlabs/client-tools/services.conf
      ]
    end

    def self.get_excluded_paths(scope)
      excluded_paths = {
        code: %w[],
        config: %w[
          /etc/puppetlabs/code
          /etc/puppetlabs/code-staging
          /etc/puppetlabs/console-services/conf.d/secrets
          /etc/puppetlabs/infra-assistant/conf.d/secrets
          /etc/puppetlabs/puppet/ssl
          /etc/puppetlabs/orchestration-services/conf.d/secrets
          /etc/puppetlabs/workflow-service/conf.d/secrets
          /opt/puppetlabs/server/data/code-manager/git
          /opt/puppetlabs/server/data/orchestration-services/data-dir
          /opt/puppetlabs/server/data/orchestration-services/code
          /opt/puppetlabs/server/data/puppetserver/filesync/storage
          /opt/puppetlabs/server/data/puppetserver/filesync/client
          /opt/puppetlabs/server/data/puppetdb
        ],
        certs: %w[],
        puppetdb: %w[
          /opt/puppetlabs/server/data/puppetdb/stockpile
        ]
      }

      included_paths = scope == ['config'] ? [] : get_included_paths(scope.reject { |s| s == 'config' })
      excluded_paths.select { |k, _| scope.include? k.to_s }
                    .values
                    .concat(global_excluded_paths)
                    .flatten
                    .reject { |path| included_paths.include? path }
    end

    def self.get_included_paths(scope)
      included_paths = {
        code: %w[
          /etc/puppetlabs/code
          /etc/puppetlabs/code-staging
          /opt/puppetlabs/server/data/puppetserver/filesync/storage
          /opt/puppetlabs/server/data/puppetserver/filesync/client
        ],
        config: %w[
          /etc/puppetlabs
          /opt/puppetlabs
        ],
        certs: %w[
          /etc/puppetlabs/puppet/ssl
        ],
        puppetdb: %w[
          /opt/puppetlabs/server/data/puppetdb
        ]
      }
      included_paths[:certs] << '/etc/puppetlabs/puppetserver/ca' if File.exist?('/etc/puppetlabs/puppetserver/ca')

      # If the scope includes 'config' all the other paths just end up being
      # redundant
      functional_scope = scope.include?('config') ? ['config'] : scope
      included_paths.select { |k, _| functional_scope.include? k.to_s }
                    .values
                    .flatten
    end

    def self.get_included_dbs(scope)
      included_dbs = {
        config: %w[
          pe-orchestrator
          pe-rbac
          pe-classifier
          pe-activity
          pe-inventory
          pe-hac
          pe-patching
          pe-infra-assistant
          pe-workflow
        ],
        puppetdb: %w[
          pe-puppetdb
        ]
      }
      included_dbs.select { |k, _| scope.include? k.to_s }.values.flatten
    end

    def self.format_excluded_paths(excluded_paths)
      excluded_paths.map { |excluded_path| "--exclude #{excluded_path}" if File.exist? excluded_path }.join(' ')
    end

    def self.format_included_paths(included_paths)
      included_paths.map { |included_path| included_path.to_s if File.exist? included_path }.join(' ')
    end

    def self.get_fs_size(scope)
      included_paths_args = format_included_paths(get_included_paths(scope))
      excluded_paths_args = format_excluded_paths(get_excluded_paths(scope))

      command = "du --bytes --max-depth=0 #{included_paths_args} #{excluded_paths_args}"
      stdout, stderr, status = run_command(command, combine: false)
      unless status.exitstatus.zero?
        error(_('Error calculating disk usage.'), command, stdout + "\n" + stderr)
      end
      stdout
    end

    def self.get_db_size(db)
      error(_('Failed to get database size: database not specified.')) unless db
      psql = "SELECT pg_database_size(datname) FROM pg_database WHERE datname = '#{db}'"
      psql_command = %(/opt/puppetlabs/server/bin/psql --tuples-only --no-align  --quiet --no-psqlrc -c \\"#{psql}\\")
      bash_command = %(su - pe-postgres -s /bin/bash -c "#{psql_command}")
      stdout, stderr, status = run_command(bash_command, combine: false)
      unless status.exitstatus.zero?
        error(_('Could not get database size for database %{db}') % { db: db }, bash_command, stdout + "\n" + stderr)
      end
      stdout.to_i
    end

    def self.estimate_backup_size(scopes)
      backup_size = { total_size: 0 }

      scopes.each do |scope|
        backup_size[scope] = { total_size: 0 }
        backup_size[scope][:filesystem] = get_fs_size([scope]).each_line.map do |row|
          size, dir = row.chomp.split(' ')
          size = size.to_i
          backup_size[:total_size] += size
          backup_size[scope][:total_size] += size
          { dir: dir, size: size }
        end
        included_dbs = get_included_dbs([scope])
        backup_size[scope][:database] = included_dbs.map do |db|
          db_size = get_db_size(db)
          backup_size[:total_size] += db_size
          backup_size[scope][:total_size] += db_size
          { db: db, size: db_size }
        end
      end
      backup_size
    end

    def self.backup_filesystem(dir, archive, scope_config)
      included_paths = format_included_paths(scope_config[:included_paths])
      excluded_paths = format_excluded_paths(scope_config[:excluded_paths])

      command = "tar -P #{excluded_paths} -cf #{dir}/#{archive} #{included_paths}"
      output = ''
      timeout = 2
      8.times do
        output, status = run_command(command)
        if %r{file changed as we read it}.match?(output)
          Puppet.warning("A file changed as we read it while backing up the filesystem, waiting #{timeout} seconds before trying again.")
          sleep timeout
          timeout *= 2
        elsif !status.exitstatus.zero?
          error(_('Failed to create file system backup due to error while compressing files.'), command, output)
        else
          return
        end
      end

      error(_('Failed to create file system backup.'), command, output)
    end

    def self.backup_database(db, dir, archive)
      command = "su - pe-postgres -s /bin/bash -c \"/opt/puppetlabs/server/bin/pg_dump -Fd -Z3 -j4 #{db} -f #{dir}/#{db}.bin\""
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Could not backup %{db} due to error while running pg_dump') % { db: db }, command, output)
      end
      command = "tar -P -C '#{dir}' -rf #{dir}/#{archive} #{db}.bin"
      output, status = run_command(command)
      unless status.exitstatus.zero?
        error(_('Could not add the database backup for %{db} to the archive') % { db: db }, command, output)
      end
    ensure
      # Remove the binary database backup now that it's just taking up space
      FileUtils.remove_entry(File.join(dir, "#{db}.bin")) if Dir.exist?(File.join(dir, "#{db}.bin"))
    end

    def self.generate_scope_config(options)
      included_paths = get_included_paths(options[:scope])
      excluded_paths = get_excluded_paths(options[:scope])
      included_dbs = get_included_dbs(options[:scope])

      {
        scope: options[:scope],
        included_paths: included_paths,
        excluded_paths: excluded_paths,
        included_dbs: included_dbs
      }
    end

    def self.compute_total_steps(scope, skip_encryption)
      steps = 14
      # recover_configuration isn't required unless :config is specified
      # 5 databases aren't included unless 'config' is specified
      steps -= 6 unless scope.include?('config')
      steps -= 1 unless scope.include?('puppetdb')
      steps -= 1 if skip_encryption
      steps
    end

    def self.backup(options)
      error(_('This command may only be run on the primary node.')) unless primary?(options) || options[:force]

      scope_config = generate_scope_config(options)

      if scope_config[:scope].include?('config') || scope_config[:scope].include?('puppetdb')
        error(_('This command cannot backup up the `certs` or `code` scopes on split installations.')) unless monolithic?(options) || options[:force]
      end

      timestamp = Time.now.utc.strftime('pe_backup-%Y-%m-%d_%H.%M.%S_UTC')
      logdir = File.expand_path(options[:logdir] || '/var/log/puppetlabs/pe-backup-tools')
      logfilename = timestamp + '.log'
      logfile = File.join(logdir, logfilename)
      PeBackupTools::Utils.configure_logging(logfile, options)

      gpg_check(options[:gpgkey]) unless options[:gpgkey].nil?

      # '/var/puppetlabs/backups' dir get created in the pe-backup-tools-vanagon packaging
      backup_dir = File.expand_path(options[:backup_dir] || '/var/puppetlabs/backups')
      archive = options[:backup_name] || timestamp + '.tgz'
      timings = {}
      step_printer = PeBackupTools::Utils::StepPrinter.new(compute_total_steps(scope_config[:scope], options[:gpgkey].nil?))

      validate_backup_location(backup_dir, archive)

      dir = Dir.mktmpdir('staging', backup_dir)
      # PostgreSQL needs permissions to write to the tmpdir
      FileUtils.chown_R('pe-postgres', 'pe-postgres', dir)

      validate_postgres_writable(dir) if scope_config[:scope].include?('config') || scope_config[:scope].include?('puppetdb')

      Puppet.info(_('Starting PE backup process. This does not disrupt Puppet Enterprise services.'))
      Puppet.info(_('Checking disk space for backup. This calculation may take a few minutes.'))
      backup_size = estimate_backup_size(scope_config[:scope])
      backup_size[:backup_name] = archive

      unless options[:force]
        timings['backup_space_comparison'] = PeBackupTools::Utils.benchmark do
          free_space, mount_point = PeBackupTools::Utils.extract_available_space_and_mount_point(backup_dir)
          # Since we can't easily estimate the compressed size of the archive before actually creating
          # the archive and because the archive file and compressed file will exist next to each other
          # for a brief moment, we will multiply the total size by 2 (this will always be an overestimate).
          working_space_needed = backup_size[:total_size] * 2
          humanized_free_space = PeBackupTools::Utils.humanize_size(free_space)
          humanized_working_space_needed = PeBackupTools::Utils.humanize_size(working_space_needed)

          spacemsg = _('Estimated backup size: %{backup_size}') % { backup_size: PeBackupTools::Utils.humanize_size(backup_size[:total_size]) }
          spacemsg += "\n" + _('      Estimated space needed to back up: %{space_needed}') % { space_needed: humanized_working_space_needed }
          spacemsg += "\n" + _('      Disk space available: %{free_space}') % { free_space: humanized_free_space }
          Puppet.info(spacemsg)
          if free_space < working_space_needed
            error_description = [
              _('There is not enough space in \'%{mount_point}\' to create the backup.') % { mount_point: mount_point },
              _('The directory has %{free_space} available, but the space needed to create the backup is %{space_needed}.') % { free_space: humanized_free_space, space_needed: humanized_working_space_needed },
              _('Free up some space in \'%{mount_point}\' or create your backup in a different location.') % { mount_point: mount_point }
            ].join(' ')
            error(error_description)
          end
        end
      end

      gpg_extension = options[:gpgkey].nil? ? '' : '.gpg'
      Puppet.info(_('Creating backup at %{backup_path}') % { backup_path: File.join(backup_dir, archive) + gpg_extension })
      Puppet.info(_('Log messages will be saved to %{logfile}') % { logfile: logfile })

      with_agent_disabled('puppet backup create in progress') do
        wait_for_agent_lock

        if scope_config[:scope].include?('config')
          timings['recover_configuration'] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Backing up PE related classification'))
            environment = options[:pe_environment] || Puppet[:environment]
            puppet_infrastructure_recover(environment, options)
          end
        end

        timings['file_system_backup'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Creating file system backup'))
          backup_filesystem(dir, archive, scope_config)
        end

        scope_config[:included_dbs].each do |db|
          timings["#{db.tr('-', '_')}_backup"] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Backing up the %{db} database') % { db: db })
            backup_database(db, dir, archive)
          end
        end
      end

      certname = Puppet[:certname]
      backup_metadata = {
        uncompressed_size: backup_size,
        scope_config: scope_config,
        certname: certname,
        pe_version: PeBackupTools::Utils.get_pe_version
      }
      write_metadata_file(dir, archive, backup_metadata)

      archive_sha = nil
      archive_size = nil
      timings['gzip_backup'] = PeBackupTools::Utils.benchmark do
        step_printer.step(_('Compressing archive file to backup directory'))
        archive_sha, archive_size = gzip_local_backup(dir, backup_dir, archive)
      end

      timings = timings.merge(total: PeBackupTools::Utils.get_total_runtime(timings))

      cert_conf = PeBackupTools::Utils.get_agent_cert_conf(certname)
      analytics_url = "https://#{certname}:8140/analytics/v1"
      PeBackupTools::Utils::AnalyticsClient.store_event(analytics_url, cert_conf, 'puppet-backup.create',
                                                        backup_dir: options[:backup_dir].nil? ? false : true,
                                                        backup_name: options[:backup_name].nil? ? false : true,
                                                        scope: scope_config[:scope],
                                                        force: options[:force] ? true : false,
                                                        timings: timings)

      backup_path = File.join(backup_dir, archive)

      encryption = 'no'
      if options[:gpgkey].nil?
        Puppet.info(_('Backup creation succeeded (size: %{size}, scope: %{scope}, time: %{time}, sha256: %{sha}, filename: %{backup_path}, gpg_encryption: %{encryption})') % { size: PeBackupTools::Utils.humanize_size(archive_size), scope: scope_config[:scope], time: PeBackupTools::Utils.humanize_time(timings[:total]), sha: archive_sha, backup_path: backup_path, encryption: encryption })
      else
        encrypted_file = nil
        encrypt_size   = nil
        timings['encrypt_backup'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Encypting archive file using gpg key '))
          encrypted_file, encrypt_size = encrypt_archive(backup_dir, archive, options[:gpgkey])
        end
        encryption = 'yes'
        backup_path = encrypted_file
        timings = timings.merge(total: PeBackupTools::Utils.get_total_runtime(timings))
        Puppet.info(_('Backup creation succeeded (size: %{size}, scope: %{scope}, time: %{time}, sha256: %{sha}, filename: %{encrypted_file}, gpg_encryption: %{encryption})') % { size: PeBackupTools::Utils.humanize_size(encrypt_size), scope: scope_config[:scope], time: PeBackupTools::Utils.humanize_time(timings[:total]), sha: archive_sha, encrypted_file: encrypted_file, encryption: encryption })
      end

      warning = _('This backup does not include the secret keys used to encrypt and decrypt sensitive data stored in the databases for the inventory and RBAC services. You must back up this information separately.')
      warning += "\n         /etc/puppetlabs/orchestration-services/conf.d/secrets"
      warning += "\n         /etc/puppetlabs/console-services/conf.d/secrets"
      warning += "\nAdditionally, if this is a Puppet Enterprise Advanced installation, there are secret keys for infra assistant and the workflow service that should be backed up separately at, respectively:"
      warning += "\n         /etc/puppetlabs/infra-assistant/conf.d/secrets"
      warning += "\n         /etc/puppetlabs/workflow-service/conf.d/secrets"
      Puppet.warning(warning)
      {
        command: 'create',
        filename: backup_path,
        digest: archive_sha,
        size: archive_size,
        scope: scope_config[:scope],
        runtime: timings
      }
    rescue RuntimeError => e
      error(_('Backup creation failed with: %{error}') % { error: e.message })
    ensure
      FileUtils.remove_entry(dir) if dir
    end
  end
end
