require 'puppet/util/pe_node_groups'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/service_status'
require 'puppet_x/util/ha'
require 'puppet_x/util/rbac'
require 'puppet_x/util/bolt'
require 'puppet_x/util/classification'
require 'puppet_x/util/docs'
require 'puppet_x/util/infrastructure_error'
require 'puppet_x/util/orchestrator'
require 'puppet_x/util/code_manager'
require 'puppet_x/util/stringformatter'

module PuppetX
module Puppetlabs
module Meep
module Provision

  # Maximum timeout secs to wait for a background puppet run to complete before
  # failing the enerprise_tasks::run_puppet task.
  MAX_RUN_PUPPET_BACKGROUND_TIMEOUT_SECS = 300

  # Original provisioning workflow relying solely on puppetdbsync.
  class Base
    include Puppet::Util::Colors
    include PuppetX::Puppetlabs::Meep::Util

    # The certificate name for the replica we are provisioning.
    attr_accessor :replica_certname

    # The options hash from the provision action.
    attr_accessor :options

    # The Puppet primary certificate name.
    attr_accessor :primary_certname

    # The array of PE services from /etc/puppetlabs/client/services.conf
    attr_accessor :config

    # The array of PE node roles from /etc/puppetlabs/client/services.conf
    attr_accessor :nodes_config

    # Hash of PE service information for service found running on the
    # +replica_certname+ host.
    attr_accessor :services

    # The RBAC token for accessing PE services.
    attr_accessor :rbac_token

    # Common information for constructing log file paths.
    attr_accessor :log_context

    def initialize(replica_certname, log_context, options, config: nil, nodes_config: nil, rbac_token: nil)
      self.replica_certname = replica_certname
      self.options = options
      self.primary_certname = Puppet.settings[:certname].strip
      self.config = config
      self.nodes_config = nodes_config
      self.rbac_token = rbac_token
      self.log_context = log_context
    end

    def check_flags
      if options[:enable]
        Puppet.err(_('The --enable flag may only be used with the --streaming flag.'))
        exit(1)
      end
    end

    # @return A Puppet::Util::Pe_node_groups instance initialized with :classifier
    # service information from the +services+ hash. Used to interact with
    # the classifier service API.
    def classifier
      unless @classifier
        nc_service = services[:classifier]
        @classifier = Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      end
      return @classifier
    end

    # Caches the UUIDs of each group/environment in the classifier. If
    # the given name can't be found, it attemps to refetch the UUIDs in
    # case classification has been updated with new groups/environments.
    # @return The UUID of the group/environment name.
    def id(name)
      if @group_ids.nil? || @group_ids[name].nil?
        @group_ids = fetch_ids
      end
      @group_ids[name]
    end

    def fetch_ids
      all_groups = PuppetX::Util::Classifier.get_groups(classifier)
      all_groups.map { |group| [group['name'], group['id']] }.to_h
    end

    # Load the config, nodes_config and services data.
    def prep_config
      self.config ||= PuppetX::Util::ServiceStatus.load_services_config()
      self.nodes_config ||= PuppetX::Util::ServiceStatus.load_nodes_config()
      self.services ||= PuppetX::Util::ServiceStatus.validate_and_select_services_from_config(config, 'provision', primary_certname)
      self.rbac_token ||= PuppetX::Util::RBAC.load_token(options[:token_file])
    end

    # Validate the primary and replica hosts.
    def validate
      orch = services[:orchestrator]
      unless options[:force]
        services_statuses = PuppetX::Util::ServiceStatus.services_for(primary_certname, nil, config, 5, nodes_config: nodes_config)
        PuppetX::Util::ServiceStatus.ensure_all_services_running(services_statuses)
        fs_status = services_statuses.find {|s| s[:service] == "file-sync-storage-service" }
        if fs_status[:status]["repos"]["puppet-code"]["latest_commit"].nil?
          Puppet.err(_("No commits have been made to puppet-code."))
          Puppet.err(_("Deploy with code-manager before provisioning by running `puppet code deploy <environment>`."))
          exit(1)
        end

        if replica_certname == primary_certname
          Puppet.err(_("%{certname} is already provisioned as the primary.") % { certname: replica_certname })
          exit(1)
        end

        existing_role = PuppetX::Util::ServiceStatus.node_role(nodes_config, replica_certname)
        if existing_role && existing_role != 'primary_master_replica'
          Puppet.err(_("%{certname} is already provisioned with the role %{role}.") % { certname: replica_certname, role: existing_role })
          exit(1)
        end

        unless PuppetX::Util::Orchestrator.node_in_orch_inventory(orch, rbac_token, replica_certname)
          Puppet.err(_("The node %{certname} is not connected to the primary via PCP.") % { certname: replica_certname } + " " +
                     _("Check that you are using the certname of the replica node, which may be different from the hostname.") + " " +
                     _("Also verify that orchestration services are enabled on the primary and that the PXP agent is running on %{certname}.") % { certname: replica_certname })
          exit(1)
        end

        env = options[:pe_environment] || 'production'
        replica_agent_version = PuppetX::Util::HA.get_host_agent_version(replica_certname, orch, rbac_token, env, log_context)
        primary_agent_version = Facter.value('aio_agent_build')

        if replica_agent_version != primary_agent_version
          Puppet.err(_("%{certname} has agent version %{replica_version} but the primary has version %{primary_version}") % { certname: replica_certname,  replica_version: replica_agent_version, primary_version: primary_agent_version})
          exit(1)
        end
      end      

      return true
    end

    # Ensure that the PE HA Master and PE HA Replica groups exist.
    # Ensure that the PE HA Master node group has the proposed replica
    # added so that permissions will be modified on the primary allowing
    # the replica access.
    def classify
      PuppetX::Util::Classification.provision_replica(classifier, primary_certname, replica_certname)

      Puppet.notice(_("Updated classification"))
    end

    def run_puppet_on(certname, description:)
      exact_log_context = log_context.step("#{description}_#{certname}")
      _invoke_task(
        'enterprise_tasks::run_puppet',
        nodes: [certname],
        params: { 'max_timeout' => PuppetX::Puppetlabs::Meep::Provision::MAX_RUN_PUPPET_BACKGROUND_TIMEOUT_SECS },
        task_log_context: exact_log_context
      )
    end

    # Enforce configuration changes on the primary and replica
    # by running Puppet through the Orchestrator.
    def run_puppet
      run_puppet_on(primary_certname, description: 'first_primary_run')

      PuppetX::Util::CodeManager.kick_file_sync_ssl()
      PuppetX::Util::CodeManager.kick_file_sync_confd()

      begin
        run_puppet_on(replica_certname, description: 'replica_run')
      ensure
        run_puppet_on(primary_certname, description: 'second_primary_run')
      end
    end

    def pin_to_pe_infra_agent
      Puppet.notice(_("Pinning #{replica_certname} to PE Infrastructure Agent"))
      PuppetX::Util::Classifier.pin_node_to_group(classifier, 'PE Infrastructure Agent', id('PE Infrastructure Agent'), replica_certname)
      run_puppet_on(replica_certname, description: 'pin_to_pe_infra_agent')
    end

    def unpin_from_pe_infra_agent
      Puppet.notice(_("Unpinning #{replica_certname} from PE Infrastructure Agent, since it should now be automatically classified."))
      PuppetX::Util::Classifier.unpin_node_from_group(classifier, 'PE Infrastructure Agent', id('PE Infrastructure Agent'), replica_certname)
    end

    # After a successful provision, inform the user and provide
    # instructions for tracking the status of the provision,
    # and subsequently enabling.
    def closing_text
      pdb_sync_link = PuppetX::Util::Docs.link(:PDB_SYNC_LINK)

      Puppet.info(_("A replica has been provisioned."))
      Puppet.notice(
          _("Services are syncing.") + " " + _("If you have a large PuppetDB instance, consult the documentation at %{link} to learn approaches for speeding up the process.") % { link: pdb_sync_link }
      )
      Puppet.notice(
        _("To track sync progress, enter the following command:\n") +
        "    puppet infrastructure status --host %{certname}" % { certname: replica_certname }
      )
      Puppet.notice(
        _("Once the sync is complete, enter the following command on the primary so that the replica will be ready to handle a failover from the primary.\n") +
        "    puppet infrastructure enable replica %{certname}" % { certname: replica_certname }
      )
      Puppet.notice(_("If you run this command before the replica has fully synchronized, the replica will not have all the necessary information to replace the primary during a failover."))
    end

    def _invoke_task(task, nodes:, params: {}, task_log_context: log_context)
      PuppetX::Puppetlabs::Meep::Provision::Base.invoke_task(
        task,
        orch_config: services[:orchestrator],
        rbac_token: rbac_token,
        display_scope: nil,
        nodes: nodes,
        params: params,
        log_context: task_log_context,
        allow_empty: false
      )
    end

    def self.invoke_task(task, orch_config:, rbac_token:, display_scope:, nodes:, params:, log_context:, allow_empty:)
      scope = case nodes
              when Hash
                nodes
              else
                { 'nodes' => nodes }
              end
      node_items = PuppetX::Util::Orchestrator.run_task(
        orch_config,
        scope: scope,
        display_scope: display_scope,
        rbac_token: rbac_token,
        task: task,
        params: params,
        log_context: log_context,
        allow_empty: allow_empty,
      )
      # Ideally, we should standardize our tasks so it always returns a
      # 'success' boolean and an 'output' hash, but we aren't quite there yet.
      if node_items != :skipped
        Puppet.debug(_("#{task} task output:"))
        node_items.each do |node|
          node_name = node['name']
          Puppet.debug("Finished #{node_name}")
          Puppet.debug(node['result'].pretty_inspect)
        end
      end

      # If Orchestrator#run_task failed, it will have raised an error, and workflow will
      # halt with that exception.
      return node_items
    end

    # Wraps a block, disabling the puppet-agent on the given nodes
    # before yielding. Ensures that the agent is renabled on the nodes
    # regardless of the success of the yielded function. Does not re-enable
    # agents if we detect that they were already disabled prior to this call
    # (agent disabling may be nested by other plans/workflows).
    #
    # @param nodes [Array<String>] certnames of the nodes to disable the
    # puppet-agent on.
    def _with_agent_disabled_on(nodes, &block)
      PuppetX::Puppetlabs::Meep::Provision::Base.with_agent_disabled_on(
        nodes,
        orch_config: services[:orchestrator],
        rbac_token: rbac_token,
        log_context: log_context,
        &block
      )
    end

    def self.with_agent_disabled_on(nodes, orch_config:, rbac_token:, log_context:, &block)
      Puppet.debug(_("Disabling agent on %{nodes}") % { nodes: nodes })
      nodes_to_disable = Array(nodes)
      node_items = []
      begin
        node_items = invoke_task(
          'enterprise_tasks::disable_agent',
          nodes: nodes_to_disable,
          orch_config: orch_config,
          rbac_token: rbac_token,
          display_scope: nil,
          params: {},
          log_context: log_context,
          allow_empty: false
        )
        yield
      ensure
        nodes_to_reenable = nodes_to_disable.filter do |cert|
          node_item = node_items.find(-> {{}}) { |ni| ni['name'] == cert }
          result = node_item['result'] || {}
          already_disabled = result['already_disabled']
          if already_disabled.nil?
            Puppet.warning(_("The enterprise_task::disable_agent task did not return a result for '%{certname}'. Since we do not know whether the agent was already disabled, no action is being taken to reenable it. Check the state of the puppet-agent on '%{certname}' manually.") % { certname: cert })
            false
          else
            Puppet.notice(_("'%{certname}' was already disabled; enable_agent skipped.") % { certname: cert }) if already_disabled
            # Only reenable if the node was not already disabled, as these
            # operations can be nested, either within provision, or within something
            # else (a Bolt plan, for example) that is calling provision.
            !already_disabled
          end
        end
        if !nodes_to_reenable.empty?
          Puppet.debug(_("Enabling agent on %{nodes}") % { nodes: nodes_to_reenable })
          invoke_task(
            'enterprise_tasks::enable_agent',
            nodes: nodes_to_reenable,
            orch_config: orch_config,
            rbac_token: rbac_token,
            display_scope: nil,
            params: {},
            log_context: log_context,
            allow_empty: false
          )
        end
      end
    end

    def run
      check_flags
      prep_config
      validate
      _with_agent_disabled_on([primary_certname, replica_certname]) do
        pin_to_pe_infra_agent
        classify
        run_puppet
        unpin_from_pe_infra_agent
      end
      closing_text
      return nil
    end
  end

end
end
end
end
