require 'puppet_x/puppetlabs/tune/inventory'
require 'puppet_x/puppetlabs/tune/query'

module PuppetX
  module Puppetlabs
    class Tune

      # Provides the configuration status of optionally enabled services like
      # the patching service.
      #
      # To be clear, this class does not check the systemctl service status or
      # process state, it is merely testing whether the service has been configured
      # to be enabled or disabled so that Tune can judge what resources to assign it,
      # if any.
      #
      # Lookups via hiera/classifier/puppetdb are performed lazily for each
      # combination of +certname+, +service+ passed to the +enabled?+ function.
      class OptionalServices

        # Provides canonical list of optional services, the +parameter+ that defines
        # whether or not a given services is disabled, the +logic+ for whether
        # a truthy value indicates services is :enabled or :disabled, and what
        # the +default+ state should be assumed to be if no configuration is
        # found.
        SERVICE_DEFINITIONS = {
          patching: {
            parameter: 'puppet_enterprise::profile::patching_service::disable_service',
            logic: :disabled,
            default: false,
          }.freeze,
          infra_assistant: {
            parameter: 'puppet_enterprise::profile::infra_assistant::disable_service',
            logic: :disabled,
            default: false,
          }.freeze,
          workflow_service: {
            parameter: 'puppet_enterprise::profile::workflow_service::disable_service',
            logic: :disabled,
            default: false,
          }.freeze
        }.freeze

        attr_reader :querier, :inventory

        def initialize(querier, inventory)
          @querier = querier
          @inventory = inventory
          @service_configuration = {}
        end

        def services
          SERVICE_DEFINITIONS.keys
        end

        def get_valid_service_key(service)
          service_key = service.to_s.downcase.to_sym
          if !services.include?(service_key)
            raise(ArgumentError, _("Tune: Unknown service '%{service}'. The only known optional services are: %{services}." % { service: service, services: services }))
          end
          service_key
        end

        # Check whether given service is enabled for given certname host.
        #
        # @parameter certname The certname of the host we are performing
        # hiera/classifier/puppetdb lookups for.
        # @parameter service String or symbol that must match a SERVICE_DEFINITIONS key.
        # @return [Boolean] True if we can find configuration in hiera, the
        # classifier, or puppetdb resources indicating the service is enabled
        # per the :parameter in its SERVICE_DEFINITIONS record for this certname.
        def enabled?(certname, service)
          service_configuration(certname, service)
        end

        def classname_from_parameter(parameter)
          classname_elements = parameter.split('::').tap(&:pop)
          full_classname = classname_elements.map { |e| e.capitalize }.join('::')
        end

        private

        def query_service_enabled(certname, service_parameter)
          # See PuppetX::Puppetlabs::Tune::Query#hiera_classifier_settings
          overrides, _ = querier.hiera_classifier_settings(certname, [service_parameter])
          state = overrides.dig('params', service_parameter)
          if state.nil?
            full_classname = classname_from_parameter(service_parameter)
            class_parameters = querier.query_puppetdb_node_class_resource_parameters(certname, full_classname)
            state = class_parameters[service_parameter]
          end
          state
        end

        def get_service_configuration_state(certname, service_key)
          if @querier.nil?
            @inventory.optional_services_enabled.include?(service_key.to_s)
          else
            definition = SERVICE_DEFINITIONS[service_key]
            configuration_state = query_service_enabled(certname, definition[:parameter])
            case configuration_state
            when nil
              definition[:default]
            else
              definition[:logic] == :disabled ?
                !configuration_state :
                !!configuration_state # ensure Boolean
            end
          end
        end

        def service_configuration(certname, service)
          service_key = get_valid_service_key(service)
          if @service_configuration.dig(certname, service_key).nil?
            @service_configuration[certname] ||= {}
            @service_configuration[certname][service_key] =
              get_service_configuration_state(certname, service_key)
          end
          @service_configuration[certname][service_key]
        end
      end
    end
  end
end
