# This class sets up puppet server and related services. For more information, see the [README.md](./README.md)
#
# @param allow_unauthenticated_ca [Boolean] True allows unauthenticated access, by default. False requires authentication on the certificate endpoints.
# @param allow_unauthenticated_status [Boolean] True allows unauthenticated access, by default. False requires authentication on the puppetserver status-service endpoint.
# @param ca_host [String] The hostname of the node acting as a certificate authority.
# @param ca_port [Integer] The port the CA service is listening on.
# @param console_host [String] The hostname of the node acting as the PE Console.
# @param enable_ca_proxy [Boolean] Enable the internal PE CA proxy that will forward agent CA requests to the @ca_host
# @param dashboard_port [Integer] *Deprecated* The port the PE console is listening on.
# @param puppetdb_host Array[String] The hostname of the PuppetDB host.
# @param submit_catalog_edges [Boolean] Flag to control whether catalog resource edges are included in catalogs submitted to PuppetDB.
# @param puppetdb_port Array[Integer] The port the PuppetDB host is listening on.
# @param console_server_certname [String] The name of the console's SSL certificate.
# @param master_of_masters_certname [String] The certname of the primary
# @param certname [String] The name of the the node's SSL certificate.
# @param classifier_host [String] The hostname of the PE Classifier.
# @param classifier_port [Integer] The port the PE Classifter is listening on.
# @param classifier_url_prefix [String] What to prefix to URLs used with the classifier.
# @param console_client_certname [String]  The name of the certificate to use when connecting to the PE Console.
# @param classifier_client_certname [String] The name of the certificate to use when connecting to the PE Classifier.
# @param localcacert [String] Certificate of the CA to use when verifying certificates.
# @param java_args [Hash] Key-value pairs describing the java arguments to be passed when starting the service.
# @param disable_string_deduplication [Boolean] When using the default G1 garbage collector, this disables
#        the use of string deduplication for optimization.
# @param metrics_server_id [String] Unique identifier for this server in the metrics backend.
# @param metrics_jmx_enabled [Boolean] Flag to enable JMX on metrics.
# @param metrics_graphite_enabled [Boolean] Flag to enable logging metrics to Graphite.
# @param metrics_graphite_host [String] The name of the Graphite server to log metrics to.
# @param metrics_graphite_port [Integer] the port the Graphite server is listening on.
# @param metrics_graphite_update_interval_seconds [Integer] How often to write metrics to the Graphite server.
# @param metrics_puppetserver_metrics_allowed [Array] A whitelist of metric names to be sent to the enabled reporters.
# @param profiler_enabled [Boolean] Flag to enable the profiler.
# @param file_sync_enabled [Variant[Boolean, Enum['automatic']]] Flag to enable file sync.
# @param cert_data_discard_history [Boolean] Discard old cert data
# @param manage_symlinks [Boolean] Flag to enable creation of convenience links
# @param r10k_remote [String] The git url for the pe_r10k configuration
# @param r10k_private_key [String] The rugged private key path for pe_r10k configuration
# @param r10k_postrun [String] An optional command that will be run after r10k finishes deploying environments. The command must be an array of strings that will be used as an argument vector.
# @param r10k_git_provider [String] The git provider to use for pe_r10k configuration, either 'rugged' or 'shellgit'
# @param r10k_known_hosts Array[Hash] The host names, types, and public keys that r10k should trust.
#                                     The format in the console should be [{"name": "github.com", "type": "ecdsa-sha2-nistp256", "key": "AAAAE2V...."}, ...]
# @param code_manager_auto_configure [Boolean] Configure code-manager based on r10k params
# @param code_manager_certname [String] The name of the code manager's SSL certificate.
# @param facts_terminus [String] The terminus to use when agents submit facts
# @param environmentpath [String] Set environmentpath in puppet.conf
# @param app_management [Boolean] Deprecated setting; app_management is always enabled
# @param code_id_command [String] The absolute path to a script to run to generate the code_id for an environment.
# @param code_content_command [String] The absolute path to a script to run to retrieve a file from an environment for a given code_id.
# @param codedir [String] Path to codedir for puppetserver and for puppet.conf setting
# @param check_for_updates [Boolean] Enable/disable update checking on puppetserver start-up.
# @param send_analytics_data [Boolean] Enable/disable telemetry reporting on puppetserver start-up.
# @param replication_mode ['source', 'replica', 'none'] Which kind of replica this server is, if any.
# @param ssl_listen_port [Integer] What port puppetserver should listen on
# @param provisioned_replicas [Array[String]] Array of certnames for provisioned replicas; these will be given access to the ca-data file sync repo.
# @param ha_enabled_replicas [Array[String]] Array of certnames of enabled replicas; these will be used as pcp contollers.
# @param connect_timeout_milliseconds [Optional[Integer]] The amount of time an outbound HTTP connection will wait to connect before giving up
# @param enable_metrics_collection [Boolean] Enable/disable collection of Puppet metrics.
# @param enable_system_metrics_collection [Boolean] Enable/disable collection of system metrics.
# @param host_action_collector_port ssl port for host_action_collector
# @param host_action_collector_host hostname for host_action_collector
# @param enable_patching_service [Boolean] Whether the patching service should be
# enabled on the primary.
# @param enable_infra_assistant [Boolean] Whether the infra assistant service should be
# enabled on the primary.
# @param infra_assistant_enable_mcp_server [Boolean] Whether to enable the MCP (Model Context Protocol)
# server for enhanced AI capabilities in the infra assistant.
# @param enable_workflow_service [Boolean] Whether the workflow service should be
# enabled on the primary.
class puppet_enterprise::profile::master(
  Boolean $allow_unauthenticated_ca                       = true,
  Boolean $allow_unauthenticated_status                   = $puppet_enterprise::allow_unauthenticated_status,
  $ca_host                                                = $puppet_enterprise::certificate_authority_host,
  $ca_port                                                = $puppet_enterprise::certificate_authority_port,
  $certname                                               = $facts['clientcert'],
  $classifier_client_certname                             = $puppet_enterprise::console_host,
  $classifier_host                                        = $puppet_enterprise::console_host,
  $classifier_port                                        = $puppet_enterprise::api_port,
  $classifier_url_prefix                                  = $puppet_enterprise::classifier_url_prefix,
  $console_client_certname                                = $puppet_enterprise::console_host,
  $console_host                                           = $puppet_enterprise::console_host,
  $console_server_certname                                = $puppet_enterprise::console_host,
  $orchestrator_client_certname                           = $puppet_enterprise::puppet_master_host,
  $master_of_masters_certname                             = $puppet_enterprise::puppet_master_host,
  $dashboard_port                                         = undef,
  Boolean $enable_ca_proxy                                = true,
  $enable_future_parser                                   = undef,
  $facts_terminus                                         = 'puppetdb',
  Hash $java_args                                         = puppet_enterprise::calculate_puppetserver_java_args(),
  Boolean $disable_string_deduplication                   = $puppet_enterprise::disable_string_deduplication,
  $localcacert                                            = $puppet_enterprise::params::localcacert,
  Optional[String] $trusted_infra_cacert                  = undef,
  String $java_keystore_passwd                            = 'changeit',
  Boolean $manage_symlinks                                = $puppet_enterprise::manage_symlinks,
  $metrics_enabled                                        = undef,
  $metrics_graphite_enabled                               = false,
  $metrics_graphite_host                                  = 'graphite',
  $metrics_graphite_port                                  = 2003,
  $metrics_graphite_update_interval_seconds               = 60,
  $metrics_puppetserver_metrics_allowed                   = undef,
  $metrics_jmx_enabled                                    = true,
  $metrics_server_id                                      = $facts['networking']['hostname'],
  $profiler_enabled                                       = true,
  Variant[Boolean, Enum['automatic']] $file_sync_enabled  = 'automatic',
  Boolean $cert_data_discard_history                      = true,
  Array[String]    $puppetdb_host                         = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer]   $puppetdb_port                         = $puppet_enterprise::puppetdb_ports_array,
  Boolean $submit_catalog_edges                           = false,
  Optional[String] $r10k_remote                           = undef,
  Optional[String] $r10k_private_key                      = undef,
  Optional[String] $r10k_proxy                            = undef,
  Optional[Array]  $r10k_postrun                          = undef,
  Array[Hash] $r10k_known_hosts                           = [],
  Optional[Enum['rugged', 'shellgit']] $r10k_git_provider = undef,
  Boolean $code_manager_auto_configure                    = false,
  String $code_manager_certname                           = $certname,
  Optional[Boolean] $versioned_deploys                    = undef,
  Optional[String] $environmentpath                       = undef,
  $app_management                                         = undef,
  Optional[String] $code_id_command                       = undef,
  Optional[String] $code_content_command                  = undef,
  Optional[String] $codedir                               = undef,
  Boolean $check_for_updates                              = true,
  Boolean $send_analytics_data                            = $puppet_enterprise::send_analytics_data,
  Puppet_enterprise::Replication_mode $replication_mode   = 'none',
  Array[String] $provisioned_replicas                     = [],
  Array[String] $ha_enabled_replicas                      = $puppet_enterprise::ha_enabled_replicas,
  Integer $ssl_listen_port                                = $puppet_enterprise::puppet_master_port,
  Optional[Integer] $connect_timeout_milliseconds         = undef,
  Boolean $enable_metrics_collection                      = $puppet_enterprise::enable_metrics_collection,
  Boolean $enable_system_metrics_collection               = $puppet_enterprise::enable_system_metrics_collection,
  Integer $host_action_collector_port                     = $puppet_enterprise::host_action_collector_port,
  String  $host_action_collector_host                     = $puppet_enterprise::host_action_collector_host,
  Boolean $enable_patching_service                        = false,
  Boolean $enable_infra_assistant                         = false,
  Boolean $infra_assistant_enable_mcp_server              = false,
  Boolean $enable_workflow_service                        = false,
) inherits puppet_enterprise {
  # This brings in the license class to ensure that the file, if present,
  # is readable by the pe-puppet user in a split install for purposes of analytics
  # reporting.
  include puppet_enterprise::license

  include puppet_enterprise::master::meep
  include puppet_enterprise::master::installer_bolt

  $compiling_server_version = pe_compiling_server_version()
  $compiling_server_aio_build = pe_compiling_server_aio_build()
  $_aio_agent_build = $facts['aio_agent_build']
  if pe_compile_master() and ($compiling_server_aio_build != $_aio_agent_build) {
    fail("This compiler has a PE version of '${pe_current_server_version()}' and an aio puppet-agent version of '${_aio_agent_build}', while the primary has version '${compiling_server_version}' and an aio puppet-agent version of '${compiling_server_aio_build}'. Please ensure that the PE versions are consistent across all infrastructure nodes by following the LEI upgrade documentation.")
  }

  $confdir = '/etc/puppetlabs/puppet'
  $puppet_default_codedir = '/etc/puppetlabs/code'
  $versioned_deploys_codedir = '/etc/puppetlabs/puppetserver/code'
  $_versioned_deploys = $versioned_deploys ? {
    false   => false,
    default => true,
  }

  $file_sync_decision = $file_sync_enabled? {
    'automatic' => $code_manager_auto_configure,
    default     => $file_sync_enabled,
  }

  if !pe_compile_master($replication_mode) {
    Package <| tag == 'pe-installer-packages' |>
    Package['pe-installer'] -> Package['pe-modules']
  }

  Pe_ini_setting {
    ensure  => present,
    path    => "${confdir}/puppet.conf",
    section => 'master',
    notify  => Service['pe-puppetserver'],
  }

  # In ankeny we set this in master we need to ensure it's absent
  pe_ini_setting { 'deprecated puppetserver puppetconf app_management' :
    ensure  => absent,
    setting => 'app_management',
    section => 'master',
  }

  # In Puppet 5 app_management is always on and the setting is deprecated
  pe_ini_setting { 'puppetserver puppetconf app_management' :
    ensure  => absent,
    setting => 'app_management',
    section => 'main',
  }

  if $enable_ca_proxy {
    # A reverse proxy between the puppet server and the ca will be configured,
    # allowing agents to have their ca traffic silently forwarded
    $ca_namespace   = 'puppetlabs.enterprise.services.reverse-proxy.reverse-proxy-ca-service'
    $ca_service     = 'reverse-proxy-ca-service'
    $ca_conf_ensure = file
    $ca_setting_ensure = present
  }
  else {
    # We disable the certificate-authority-service
    # Relevant resources are collected and overridden in the ca profile
    $ca_namespace   = 'puppetlabs.services.ca.certificate-authority-disabled-service'
    $ca_service     = 'certificate-authority-disabled-service'
    $ca_conf_ensure = absent
    $ca_setting_ensure = absent
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { 'certificate-authority-service' :
    container => 'puppetserver',
    namespace => $ca_namespace,
    service   => $ca_service,
    require   => Package['pe-puppetserver'],
    notify    => Service['pe-puppetserver'],
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { 'host-action-collector-service' :
    container => 'puppetserver',
    namespace => 'puppetlabs.host-action-collector.service',
    service   => 'host-action-collector-service',
  }

  $puppetserver_confdir = '/etc/puppetlabs/puppetserver/conf.d'

  Pe_hocon_setting {
    ensure  => present,
    require => Package['pe-puppetserver'],
    notify  => Service['pe-puppetserver'],
  }

  # Uses
  #   $metrics_graphite_host
  #   $metrics_graphite_port
  #   $metrics_graphite_update_interval_seconds
  # Global graphite host setting
  pe_hocon_setting { 'metrics.reporters.graphite.host':
    path    => "${puppetserver_confdir}/metrics.conf",
    setting => 'metrics.reporters.graphite.host',
    value   => $metrics_graphite_host,
    require => File["${puppetserver_confdir}/metrics.conf"],
  }
  # Global graphite port setting
  pe_hocon_setting { 'metrics.reporters.graphite.port':
    path    => "${puppetserver_confdir}/metrics.conf",
    setting => 'metrics.reporters.graphite.port',
    value   => $metrics_graphite_port,
    require => File["${puppetserver_confdir}/metrics.conf"],
  }
  # Global setting for how often to send metrics
  pe_hocon_setting { 'metrics.reporters.graphite.update-interval-seconds':
    path    => "${puppetserver_confdir}/metrics.conf",
    setting => 'metrics.reporters.graphite.update-interval-seconds',
    value   => $metrics_graphite_update_interval_seconds,
    require => File["${puppetserver_confdir}/metrics.conf"],
  }

  # Uses
  #   $ca_host
  #   $ca_port
  #   $certname
  #   $localcacert
  file { "${puppetserver_confdir}/ca.conf":
    ensure  => $ca_conf_ensure,
    require => Package['pe-puppetserver'],
  }

  pe_hocon_setting{ 'certificate-authority.proxy-config.proxy-target-url':
    ensure  => $ca_setting_ensure,
    path    => "${puppetserver_confdir}/ca.conf",
    setting => 'certificate-authority.proxy-config.proxy-target-url',
    value   => "https://${ca_host}:${ca_port}",
    require => Package['pe-puppetserver'],
  }
  pe_hocon_setting{ 'certificate-authority.proxy-config.ssl-opts.ssl-cert':
    ensure  => $ca_setting_ensure,
    path    => "${puppetserver_confdir}/ca.conf",
    setting => 'certificate-authority.proxy-config.ssl-opts.ssl-cert',
    value   => "/etc/puppetlabs/puppet/ssl/certs/${certname}.pem",
    require => Package['pe-puppetserver'],
  }
  pe_hocon_setting{ 'certificate-authority.proxy-config.ssl-opts.ssl-key':
    ensure  => $ca_setting_ensure,
    path    => "${puppetserver_confdir}/ca.conf",
    setting => 'certificate-authority.proxy-config.ssl-opts.ssl-key',
    value   => "/etc/puppetlabs/puppet/ssl/private_keys/${certname}.pem",
    require => Package['pe-puppetserver'],
  }
  pe_hocon_setting{ 'certificate-authority.proxy-config.ssl-opts.ssl-ca-cert':
    ensure  => $ca_setting_ensure,
    path    => "${puppetserver_confdir}/ca.conf",
    setting => 'certificate-authority.proxy-config.ssl-opts.ssl-ca-cert',
    value   => $localcacert,
    require => Package['pe-puppetserver'],
  }

  file { "${puppetserver_confdir}/host-action-collector.conf":
    ensure  => present,
    require => Package['pe-puppetserver'],
  }

  pe_hocon_setting { 'host-action-collector.certs.ssl-key':
    ensure  => present,
    path    => "${puppetserver_confdir}/host-action-collector.conf",
    setting => 'host-action-collector.certs.ssl-key',
    value   => "/etc/puppetlabs/puppet/ssl/private_keys/${certname}.pem",
    require => Package['pe-puppetserver'],
  }

  pe_hocon_setting { 'host-action-collector.certs.ssl-ca-cert':
    ensure  => present,
    path    => "${puppetserver_confdir}/host-action-collector.conf",
    setting => 'host-action-collector.certs.ssl-ca-cert',
    value   => $localcacert,
    require => Package['pe-puppetserver'],
  }

  pe_hocon_setting { 'host-action-collector.certs.ssl-cert':
    ensure  => present,
    path    => "${puppetserver_confdir}/host-action-collector.conf",
    setting => 'host-action-collector.certs.ssl-cert',
    value   => "/etc/puppetlabs/puppet/ssl/certs/${certname}.pem",
    require => Package['pe-puppetserver'],
  }

  pe_hocon_setting { 'host-action-collector.storage-location':
    ensure  => present,
    path    => "${puppetserver_confdir}/host-action-collector.conf",
    setting => 'host-action-collector.storage-location',
    value   => '/var/log/puppetlabs/puppetserver/data',
    require => Package['pe-puppetserver'],
  }


  # if we are on the primary, route actions to the action collection service, if not, route
  # them to the primary's proxy interface which will in turn forward them.

  if ($certname == $master_of_masters_certname) {
    $host_action_collector_url = "https://${host_action_collector_host}:${host_action_collector_port}"
  } else {
    $host_action_collector_url = "https://${master_of_masters_certname}:${ssl_listen_port}/client-events"
  }

  pe_hocon_setting { "host-action-collector.collector-host-url":
    ensure  => present,
    path    => "${puppetserver_confdir}/host-action-collector.conf",
    setting => 'host-action-collector.collector-host-url',
    value   => $host_action_collector_url,
    require => Package['pe-puppetserver'],
  }

  # Updating puppet.conf node terminus
  if ! pe_empty($classifier_host) {
    class { 'puppet_enterprise::profile::master::classifier' :
      classifier_host       => $classifier_host,
      classifier_port       => $classifier_port,
      classifier_url_prefix => $classifier_url_prefix,
      notify                => Service['pe-puppetserver'],
      require               => Package['pe-puppetserver'],
    }
  }

  # Only copy the hostcrl if this is a puppet agent run.
  # This is needed for working with a puppet apply that also
  # generates the certs in the same run.
  if pe_servername() {
    # Copy the crl from the ca if we're a secondary server
    # and if we're the ca just move it to the hostcrl as the canonical location
    $_crl_content = file($::settings::cacrl, $::settings::hostcrl, '/dev/null')
    if pe_empty($_crl_content) {
      warning("CRL file not found on ${::servername}")
    } else {
      file { $puppet_enterprise::params::hostcrl:
        ensure  => file,
        owner   => 'pe-puppet',
        group   => 'pe-puppet',
        mode    => '0644',
        content => $_crl_content,
        require => Package['pe-puppetserver'],
      }
    }
  }

  $private_keys_dir = "${puppet_enterprise::params::ssl_dir}/private_keys"

  file { $private_keys_dir :
    ensure  => directory,
    owner   => 'pe-puppet',
    group   => 'pe-puppet',
    mode    => '0640',
    require => Package['pe-puppetserver'],
  }

  file { "${private_keys_dir}/${facts['clientcert']}.pem":
    ensure  => file,
    owner   => 'pe-puppet',
    group   => 'pe-puppet',
    mode    => '0640',
    require => Package['pe-puppetserver'],
    notify  => Service['pe-puppetserver'],
  }

  puppet_enterprise::set_owner_group_permissions{ $private_keys_dir :
    file_mode   => '0640',
    owner       => 'pe-puppet',
    group       => 'pe-puppet',
    target_type => 'directory',
    dir_mode    => '0750',
    require     => [ Package['pe-puppetserver'],
                     File["${puppet_enterprise::params::ssl_dir}/private_keys/${facts['clientcert']}.pem"]]
  }

  # PMT access to pe_only content
  pe_ini_setting { 'module_groups' :
    section => 'main',
    setting => 'module_groups',
    value   => 'base+pe_only',
  }

  # custom codedir, use it always
  if $codedir {
    $real_codedir = $codedir
    $codedir_ensure = 'directory'
    $codedir_setting_ensure = 'present'

  } elsif $_versioned_deploys and ($code_manager_auto_configure or $file_sync_decision) {
    $real_codedir = $versioned_deploys_codedir
    $codedir_ensure = 'present'
    $codedir_setting_ensure = 'present'

    # Manage the default codedir so we can link to it when toggling $_versioned_deploys
    file { $puppet_default_codedir:
      ensure  => directory,
      owner   => 'pe-puppet',
      group   => 'pe-puppet',
      mode    => '0740',
      recurse => false,
      require => Package['pe-puppetserver'],
    }
  } else {
    # Fall back to Puppet's defaults
    $real_codedir = $puppet_default_codedir
    $codedir_ensure = 'directory'
    $codedir_setting_ensure = 'absent'
  }

  # Manage the directory that will contain the versioned code symlink
  # The package provides this directory but it is good to ensure its
  # permissions don't change regardless of versioned code settings.
  file { '/etc/puppetlabs/puppetserver':
    ensure  => directory,
    owner   => 'pe-puppet',
    group   => 'pe-puppet',
    mode    => '0740',
    recurse => false,
    require => Package['pe-puppetserver'],
  }

  file { $real_codedir:
    ensure  => $codedir_ensure,
    owner   => 'pe-puppet',
    group   => 'pe-puppet',
    mode    => '0740',
    recurse => false,
    require => Package['pe-puppetserver'],
  }

  pe_ini_setting { 'codedir_setting':
    ensure  => $codedir_setting_ensure,
    setting => 'codedir',
    value   => $real_codedir,
    section => 'main',
    require => File[$real_codedir],
  }

  if $environmentpath {
    $dangerous_envpath = $environmentpath.match(Regexp[$real_codedir])

    if $dangerous_envpath {
      warning("Do not hardcode environmentpath to include the value of the codedir. Use Puppet's interpolation capabilities instead. Found codedir ${real_codedir} in environmentpath ${environmentpath}")
    }

    pe_ini_setting { 'environmentpath_setting':
      setting => 'environmentpath',
      value   => $environmentpath,
      section => 'main',
    }
  } else {
    pe_ini_setting { 'environmentpath_setting':
      ensure  => absent,
      setting => 'environmentpath',
      section => 'main',
    }
  }


  class { 'puppet_enterprise::profile::master::auth_conf':
    console_client_certname      => $console_client_certname,
    classifier_client_certname   => $classifier_client_certname,
    orchestrator_client_certname => $orchestrator_client_certname,
    allow_unauthenticated_ca     => $allow_unauthenticated_ca,
    allow_unauthenticated_status => $allow_unauthenticated_status,
    require                      => Package['pe-puppetserver'],
    notify                       => Service['pe-puppetserver'],
  }

  class { 'puppet_enterprise::profile::master::puppetdb' :
    puppetdb_host         => $puppetdb_host,
    puppetdb_port         => $puppetdb_port,
    facts_terminus        => $facts_terminus,
    include_catalog_edges => $submit_catalog_edges,
    notify                => Service['pe-puppetserver'],
    require               => Package['pe-puppetserver'],
  }

  # When terminus configuration is going to do some kind of failover (i.e. in HA
  # configurations), we need to use a smaller connect timeout. Only do this if
  # the parameter wasn't explicitly specified.
  $is_ha = (pe_count(pe_any2array($puppetdb_host)) > 1) or (pe_count(pe_any2array($classifier_host)) > 1)
  $defaulted_connect_timeout = if $connect_timeout_milliseconds {
                                  $connect_timeout_milliseconds
                                } elsif $is_ha {
                                  5000
                                } else {
                                  undef
                                }

  class { 'puppet_enterprise::master' :
    certname                                  => $certname,
    static_files                              => {'/packages' => "${puppet_enterprise::packages_dir}/public"},
    localcacert                               => $localcacert,
    java_args                                 => $java_args,
    disable_string_deduplication              => $disable_string_deduplication,
    metrics_server_id                         => $metrics_server_id,
    metrics_jmx_enabled                       => $metrics_jmx_enabled,
    metrics_graphite_enabled                  => $metrics_graphite_enabled,
    metrics_puppetserver_metrics_allowed      => $metrics_puppetserver_metrics_allowed,
    profiler_enabled                          => $profiler_enabled,
    enable_future_parser                      => $enable_future_parser,
    manage_symlinks                           => $manage_symlinks,
    code_manager_auto_configure               => $code_manager_auto_configure,
    puppetserver_jruby_puppet_master_code_dir => $real_codedir,
    ssl_listen_port                           => $ssl_listen_port,
    connect_timeout_milliseconds              => $defaulted_connect_timeout,
  }

  # Only compilers have pcp-broker. On the primary and replica, pcp-broker
  # lives in orchestration-services.
  if (pe_compile_master($replication_mode)) {
    class { 'puppet_enterprise::master::pcp_broker':
      orchestrator_hosts => [$puppet_enterprise::puppet_master_host] + $ha_enabled_replicas
    }
  }

  # PE MODULE DEPLOYMENT
  # Puppet Enterprise supported deployments need to have a base set of modules
  # installed. These modules are not packaged using rpm or deb, but come from
  # tarball sources. Ensure that each required module is retrieved and
  # deployed.

  $sharedir = $facts['os']['family'] ? {
    'windows' => "${facts['common_appdata']}/PuppetLabs/puppet/share",
    default   => "${puppet_enterprise::server_share_dir}/puppet_enterprise",
  }

  file { $sharedir:
    ensure => directory,
    owner  => $puppet_enterprise::params::root_user,
    group  => $puppet_enterprise::params::root_group,
    mode   => '0755',
    require => Package['pe-puppet-enterprise-release'],
  }

  $public_packages_dir = "${puppet_enterprise::packages_dir}/public"

  # This directory is managed by pe_repo.  However, if this directory does not
  # exist and pe_repo is not part of the classification (during initial installation
  # puppet apply) then an error will be thrown by Puppet's fileserver mount code.
  exec { "Ensure public dir ${public_packages_dir}":
      command => "mkdir -p ${public_packages_dir}",
      path    => '/sbin/:/bin/',
      unless  => "ls ${public_packages_dir}",
      require => Package['pe-puppetserver'],
  }

  # FILESERVER
  # Allow Puppet to serve puppet-agent packages from the public fileserver set up
  # by pe_repo.
  puppet_enterprise::fileserver_conf { $puppet_enterprise::packages_mountpoint:
    mountpoint => $puppet_enterprise::packages_mountpoint,
    path       => $public_packages_dir,
    require    => Exec["Ensure public dir ${public_packages_dir}"],
  }

  #include the proxy host-action-client on the primary only
  if (!pe_compile_master($replication_mode) and $replication_mode != 'replica') {
    puppet_enterprise::trapperkeeper::bootstrap_cfg { 'client-events-service' :
      container => 'puppetserver',
      namespace => 'puppetlabs.enterprise.services.host-actions.client-events-service',
    }
  }

  pe_ini_setting { 'toggle versioned environment dirs':
    ensure  => present,
    setting => 'versioned_environment_dirs',
    value   => true,
  }

  pe_ini_setting { 'set report_configured_environmentpath':
    ensure  => present,
    setting => 'report_configured_environmentpath',
    value   => true,
  }

  if (!pe_compile_master($replication_mode) and $replication_mode != 'replica') and $enable_metrics_collection {
    include puppet_enterprise::pe_metrics
  }

  # The system class is included on all infra nodes
  if $enable_system_metrics_collection {
    include puppet_enterprise::pe_system_metrics
  }

  $send_updates_ensure = $check_for_updates ? {
    true  => absent,
    false => file,
  }

  file { '/etc/puppetlabs/puppetserver/opt-out':
    ensure  => $send_updates_ensure,
    notify  => Service['pe-puppetserver'],
    require => Package['pe-puppetserver']
  }

  $analytics_opt_out_ensure = $send_analytics_data ? {
    true  => absent,
    false => file,
  }

  file { '/etc/puppetlabs/analytics-opt-out':
    ensure  => $analytics_opt_out_ensure,
    notify  => Service['pe-puppetserver'],
    require => Package['pe-puppetserver']
  }

  # We only want one analytics service
  # Analytics has some dependencies on pe-puppetserver for now so should not be on a separate host
  if (!pe_compile_master($replication_mode) and $replication_mode != 'replica') {
    class { 'puppet_enterprise::trapperkeeper::analytics':
      container => 'puppetserver',
    }
  }

  # We don't want the host action collector service on compilers. On the replica,
  # we want to set up all the configuration and the database, but don't want
  # to actually turn on the service.
  if (!pe_compile_master($replication_mode)) {
    class { 'puppet_enterprise::master::host_action_collector':
      replication_mode   => $replication_mode,
      disable_service    => $replication_mode == 'replica',
      ssl_listen_port    => $host_action_collector_port,
    }
  }

  # Similarly we don't want the patching service / infra assistant on compilers.
  # On the primary we want it enabled based on the flag.
  # On replica we want it installed and configured, but always disabled.
  if (!pe_compile_master($replication_mode)) {
    $patching_service_enabled = ($replication_mode != 'replica') ? {
      true    => $enable_patching_service, # primary
      default => false,                    # replica
    }
    class { 'puppet_enterprise::profile::patching_service':
      disable_service    => !$patching_service_enabled,
    }
  }

  if (!pe_compile_master($replication_mode)) {
    $infra_assistant_enabled  = ($replication_mode != 'replica') ? {
      true    => $enable_infra_assistant, # primary
      default => false,                    # replica
    }
    # TODO: Review MCP server opt-in implementation following best practices
    class { 'puppet_enterprise::profile::infra_assistant':
      disable_service    => !$infra_assistant_enabled,
      enable_mcp_server  => $infra_assistant_enable_mcp_server,
    }
  }

  if (!pe_compile_master($replication_mode)) {
    $workflow_service_enabled = ($replication_mode != 'replica') ? {
      true    => $enable_workflow_service, # primary
      default => false,                    #replica
    }
    class { 'puppet_enterprise::profile::workflow_service':
      disable_service => !$workflow_service_enabled,
    }
  }

  class { 'puppet_enterprise::master::code_management':
    certname                    => $certname,
    puppetserver_codedir        => $real_codedir,
    puppet_default_codedir      => $puppet_default_codedir,
    versioned_deploys_codedir   => $versioned_deploys_codedir,
    environmentpath             => $environmentpath,
    code_manager_auto_configure => $code_manager_auto_configure,
    code_manager_certname       => $code_manager_certname,
    file_sync_enabled           => $file_sync_decision,
    versioned_deploys_enabled   => $_versioned_deploys,
    ssl_listen_port             => $ssl_listen_port,
    primary_certname            => $master_of_masters_certname,
    replication_mode            => $replication_mode,
    provisioned_replicas        => $provisioned_replicas,
    cert_data_discard_history   => $cert_data_discard_history,
    r10k_known_hosts            => $r10k_known_hosts,
    r10k_remote                 => $r10k_remote,
    git_private_key             => $r10k_private_key,
    r10k_proxy                  => $r10k_proxy,
    r10k_postrun                => $r10k_postrun,
    git_provider                => $r10k_git_provider,
    code_content_command        => $code_content_command,
    code_id_command             => $code_id_command,
  }

  # Configure r10k and Java keystore to trust an infra certificate
  if $trusted_infra_cacert {
    $trusted_cacert_ensure = file
    $pe_java_keytool_notify = Exec['pe_java_keytool_import']
    # have to allow failures of this command because the initial addition
    # would fail with "alias does not exist" otherwise
    $allowed_keytool_delete_exitcodes = [0, 1]
  } else {
    # When not managing the trusted cacert ensure that the link is absent so
    # the openssl rehash removes it from its trusted set, and that the keytool
    # exec restarts pe-puppetserver after deleting the cert
    $trusted_cacert_ensure = absent
    $pe_java_keytool_notify = Service['pe-puppetserver']
    $allowed_keytool_delete_exitcodes = 0
  }

  # Manage cacert file, rather than a symlink, so that updates to the content
  # of the cacert will notify downstream resources.
  $trusted_cacert_path = '/opt/puppetlabs/puppet/ssl/certs/pe-trusted-infra-cacert.pem'
  $keytool_alias = 'pe-trusted-infra-cacert'
  file { $trusted_cacert_path:
    ensure  => $trusted_cacert_ensure,
    content => $trusted_infra_cacert,
    owner   => 'pe-puppet',
    group   => 'pe-puppet',
    mode    => '0644',
    require => Package['pe-puppetserver'],
    # notify the keytool delete exec so that a change in the trusted certificate
    # causes the keytool's cert to be updated
    notify  => [Exec['pe_puppet_openssl_rehash'], Exec['pe_java_keytool_delete']],
  }

  exec {'pe_puppet_openssl_rehash':
    refreshonly => true,
    command     => '/opt/puppetlabs/puppet/bin/openssl rehash',
  }

  exec { 'pe_java_keytool_delete':
    refreshonly => true,
    command     => "/opt/puppetlabs/server/bin/keytool -delete -cacerts -alias ${keytool_alias} -storepass '${java_keystore_passwd}'",
    notify      => $pe_java_keytool_notify,
    returns     => $allowed_keytool_delete_exitcodes,
  }

  exec { 'pe_java_keytool_import':
    path    => $facts['path'],
    onlyif  => "test -f ${trusted_cacert_path}",
    unless  => "/opt/puppetlabs/server/bin/keytool -list -cacerts -alias ${keytool_alias}",
    command => "/opt/puppetlabs/server/bin/keytool -importcert -cacerts -trustcacerts -noprompt -storepass '${java_keystore_passwd}' -alias ${keytool_alias} -file ${trusted_cacert_path}",
    require => File[$trusted_cacert_path],
    notify  => Service['pe-puppetserver'],
  }

  include puppet_enterprise::profile::controller
}
