#!/usr/bin/env ruby
# frozen_string_literal: true

# Example task that retrieves interface information from a NETCONF device
# This demonstrates using filters to get specific configuration elements
#
# Usage:
#   bolt task run get_interfaces --targets netconf://device.example.com

require 'rexml/document'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  session.logger.info('Retrieving interface information')

  # Create a filter for interfaces
  # This example uses a generic filter that should work on many devices
  # Adjust based on your device's data model
  filter_doc = REXML::Document.new
  interfaces_elem = REXML::Element.new('interfaces')
  interfaces_elem.add(REXML::Element.new('interface'))
  filter_doc.add(interfaces_elem)
  filter = filter_doc.root

  # Get filtered configuration
  begin
    # Try to get from running config first
    result = session.get_config('running', filter)
    session.logger.info('Retrieved interface configuration')
  rescue => e
    session.logger.warn("Failed to get interfaces from config: #{e.message}")
    session.logger.info('Trying operational data instead')

    # Some devices may have interfaces in operational data
    result = session.get(filter)
    session.logger.info('Retrieved interface operational data')
  end

  # Parse interface data
  interfaces = []
  result.each do |element|
    next unless element.name == 'interfaces'
    element.elements.each('.//interface') do |intf|
      interface_data = {
        'name' => intf.elements['name']&.text,
        'description' => intf.elements['description']&.text,
        'enabled' => intf.elements['enabled']&.text || intf.elements['admin-status']&.text,
        'mtu' => intf.elements['mtu']&.text
      }.compact

      interfaces << interface_data unless interface_data.empty?
    end
  end

  session.logger.info("Found #{interfaces.length} interfaces")

  # Report the interfaces
  session.report_result({
                          'interfaces' => interfaces,
    'count' => interfaces.length
                        })
end

# Output the result as JSON for Bolt
require 'json'
puts result.to_json
