#!/usr/bin/env ruby
# frozen_string_literal: true

require 'json'

# Load the task helper to set up the Ruby load path
require_relative '../files/task_helper.rb'

# Now require the netconf library
require 'puppet_x/puppetlabs/netconf/session'

# Kill a NETCONF session
#
# This task kills a specified NETCONF session using the kill-session operation.
# According to RFC 6241, this will:
# - Abort any operations currently in process
# - Release any locks and resources associated with the session
# - Close any associated connections
# Note: Configuration changes made by the killed session are NOT rolled back.

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Get the session ID from task parameters
  session_id_to_kill = session.task_params['session_id']

  # Validate that session_id is provided
  if session_id_to_kill.nil? || session_id_to_kill.to_s.empty?
    raise ArgumentError, 'session_id parameter is required'
  end

  # Kill the specified session
  session.connection.rpc.kill_session(session_id_to_kill)

  {
    status: 'success',
    message: "Successfully killed session #{session_id_to_kill}",
    killed_session_id: session_id_to_kill,
    current_session_id: session.connection.session_id
  }
end

puts result.to_json
