require 'hocon'
require 'hocon/config_factory'
require 'hocon/parser/config_document_factory'
require 'hocon/config_value_factory'

module PuppetX
module Puppetlabs
module Meep
  # Provides a separate class for modifying the pe.conf file. Modify
  # does not use Puppet Lookup. It operates directly on the configuration file
  # via the Hocon library.
  #
  # * {Modify#set_in_pe_conf Modify#set_in_pe_conf(key,value)} sets specified
  #   entry in the pe.conf Object. May nest object keys via 'dot' pathing to
  #   address inner elements of hashes (objects). See the method documentation
  #   for details.
  #
  # Concurrency
  # -----------
  #
  # This library uses File.flock in order to allow two different processes to
  # make atomic changes to pe.conf through our setters without stepping on each
  # other.
  #
  # This makes absolutely no guarantees about any other process arbitrarily
  # writing the file, it is only intended to allow the infrastructure face to
  # coexist with other scripts that use the face, or the Config library
  # directly.
  class Modify

    # The path to the enterprise hieradata
    attr_reader :enterprise_dir

    # @param path [String] The path to the meep data file to be modified.
    def initialize(enterprise_dir)
      @enterprise_dir = File.expand_path(enterprise_dir)
    end

    def get_in_pe_conf(key)
      pe_conf_mutator._get_in_pe_conf(key)
    end

    # Directly add or update a key, value pair in the local pe.conf
    # file.  This should only be used by other infrastructure faces to
    # adjust pe.conf on the meep master.
    #
    # This method can be used to set root keys, or to set elements within a hash
    # if a compound key is provided:
    #
    #   set_in_pe_conf(
    #     "node_roles",
    #     { "pe_role::monolithic::primary_master" => ["master.net"] }
    #   )
    #
    # for example will replace the node_roles hash entirely. While:
    #
    #   set_in_pe_conf(
    #     '"node_roles"."pe_role::monolithic::primary_master"',
    #     ["master.net"]
    #   )
    #
    # will instead just change the primary_master role's array. (Take note
    # of the quoting in the key...colons are not permitted otherwise...)
    #
    # @param key [String] the name of the pe.conf key.
    # @param value [String,Array,Hash,Numeric,Boolean] the value to assign
    #   to the given key. Must be something that will convert to a valid
    #   JSON type.
    # @return [Boolean] true if successful.
    # @raise [Hocon::ConfigError::ConfigParseError] if pe.conf format is
    #   invalid Hocon.
    # @raise [Puppet::Error] if there is no pe.conf file to work with.
    def set_in_pe_conf(key, value)
      pe_conf_mutator._set_in_pe_conf(key,value)
    end

    private

    def pe_conf_mutator
      SafeMutator.new("#{self.enterprise_dir}/conf.d/pe.conf")
    end

    # This class has the core operations that modify pe.conf, with no
    # concern for locking or concurrency.
    class PeConfMutator
      attr_accessor :conf_file_path

      def initialize(conf_file_path)
        self.conf_file_path = conf_file_path
      end

      # If the key is unquoted and does not contain pathing ('.'),
      # quote to ensure that puppet namespaces are protected
      #
      # @example
      #   _quoted_hocon_key("puppet_enterprise::database_host")
      #   # => '"puppet_enterprise::database_host"'
      #
      def _quoted_hocon_key(key)
        case key
        when /^[^"][^.]+/
          # if the key is unquoted and does not contain pathing ('.')
          # quote to ensure that puppet namespaces are protected
          # ("puppet_enterprise::database_host" for example...)
          then %Q{"#{key}"}
        else key
        end
      end

      def _get_in_pe_conf(key, default = nil)
        doc = Hocon::ConfigFactory.parse_file(conf_file_path)
        hocon_key = _quoted_hocon_key(key)
        doc.has_path?(hocon_key) ?
          doc.get_value(hocon_key).unwrapped :
          default
      end

      def _set_in_pe_conf(key, value)
        pe_conf = Hocon::Parser::ConfigDocumentFactory.parse_file(conf_file_path)

        hocon_key = _quoted_hocon_key(key)

        hocon_value = case value
        when String
          # ensure unquoted string values are quoted for uniformity
          then value.match(/^[^"]/) ? %Q{"#{value}"} : value
        else Hocon::ConfigValueFactory.from_any_ref(value, nil)
        end

        pe_conf = value.kind_of?(String) ?
          pe_conf.set_value(hocon_key, hocon_value) :
          pe_conf.set_config_value(hocon_key, hocon_value)

        File.open(conf_file_path, 'w') do |fh|
          config_string = pe_conf.render
          fh.puts(config_string)
        end

        return true
      end
    end

    # A subclass of Mutator that makes use of flock to ensure that the
    # functions are atomic and can be run concurrently in separate processes.
    #
    # This should allow scripts in PE (autosigning, for example) to run without
    # fear of mangling pe.conf due to simultaneous changes from a command line
    # invocation of `puppet infrastructure provision`.
    #
    # It relies on Puppet::FileSystem.exclusive_open under the hood for locking
    # and timeout behavior.
    #
    # The default timeout implemented in this class is {SafeMutator::TIMEOUT}.
    class SafeMutator < PeConfMutator

      # Default timeout in seconds
      TIMEOUT = 30

      def self.timeout
        @timeout || TIMEOUT
      end

      # This is a convenience for testing.
      def self.timeout=(timeout)
        @timeout = timeout
      end

      def timeout
        self.class.timeout
      end

      def _set_in_pe_conf(key, value)
        _lock_and_set(conf_file_path) do
          super
        end
      end

      def locked?
        @locked
      end

      def lock!
        @locked = true
      end

      def unlock!
        @locked = false
      end

      def _lock_and_set(file)
        if locked?
          result = yield
        else
          lock!
          result = nil
          begin
            raise(Puppet::Error, "No pe.conf file found at #{conf_file_path}") unless File.exist?(conf_file_path)

            Puppet::FileSystem.exclusive_open(file, 0600, 'r', timeout) do
              result = yield
              unlock!
            end
          rescue Timeout::Error
            raise(Puppet::Error, "Unable to get a lock on #{file} within #{timeout}s.")
          end
        end

        return result
      end
    end
  end
end
end
end
