require 'net/https'
require 'puppet'
require 'digest'

module PuppetX
  module Util
    class Analytics

      PE_CONF_FILE = '/etc/puppetlabs/enterprise/conf.d/pe.conf'

      # Tracking ID for Google Analytics
      ANALYTICS_TID_DEV = "UA-80924048-4"
      ANALYTICS_TID_PROD = "UA-80924048-5"

      def self.send_analytics_event(category,action,label)
        self.check_analytics_setting if @disable.nil?
        return if @disable
        [category,action,label].each do |value|
          if value.nil? || value.empty?
            Puppet.debug(_("Arguments must not be nil or empty strings for analytics event. Analytics event not sent."))
            return
          end
        end

        category = URI.encode_www_form_component(category)
        action = URI.encode_www_form_component(action)
        label = URI.encode_www_form_component(label)

        begin
          networking = Facter.value("networking")
          mac = networking.nil? ? "" : networking["mac"]
          fqdn = Facter.value("fqdn")
          uid = Digest::SHA2.hexdigest("#{mac}#{fqdn}")
          payload = "v=1&t=event&tid=#{self.tid}&cid=#{uid}&ec=#{category}&ea=#{action}&el=#{label}"
          uri = URI.parse("http://www.google-analytics.com/collect")
          http = Net::HTTP.new(uri.host,uri.port)
          http.max_retries = 0
          http.read_timeout = 1
          http.open_timeout = 1

          request = Net::HTTP::Post.new(uri.request_uri)
          request.body = payload
          response = http.request(request)
          raise HTTPUnexpectedResponse.new(response) if response.code != "200"
          Puppet.notice(_("Sent analytics: %{category} - %{action} - %{label}") % {category: category, action: action, label: label})
        rescue SocketError
          Puppet.debug(_("Error attempting to send analytics: Could not reach google-analytics.com"))
        rescue Net::ReadTimeout, Net::OpenTimeout
          Puppet.debug(_("Error attempting to send analytics: Timeout talking to Google Analytics server"))
        rescue HTTPUnexpectedResponse => e
          Puppet.debug(_("Error attempting to send analytics: Did not receive a 200 OK response from Google Analytics. Received the following code: %{code}") % {code: e.response.code})
        rescue => e
          Puppet.debug(_("Error attempting to send analytics with the following exception: %{e}") % {e: e})
        end
      end

      def self.tid
        fqdn = Facter.value("fqdn")
        fqdn = fqdn.nil? ? "" : fqdn
        # Specs should not be sending analytics events, but including Travis here just in case
        fqdn.include?("puppetlabs.net") || fqdn.include?("puppet.com") || fqdn.include?("travis-ci") ? ANALYTICS_TID_DEV : ANALYTICS_TID_PROD
      end

      def self.check_analytics_setting
        begin
          pe_conf = Hocon.load(PE_CONF_FILE)
          send_analytics = pe_conf["puppet_enterprise::send_analytics_data"]
          if !send_analytics.nil? && !send_analytics
            Puppet.notice(_("puppet_enterprise::send_analytics_data = false. Analytics events will not be sent."))
            @disable = true
          else
            @disable = false
          end
        rescue Hocon::ConfigError
          Puppet.debug(_("Error attempting to load pe.conf to verify analytics setting. Analytics events will not be sent."))
          @disable = true
        end
      end
    end

    class HTTPUnexpectedResponse < StandardError
      attr_reader :response
      def initialize(response=nil)
        @response = response
        super
      end
    end
  end
end
