require 'puppet'
require 'puppet/http'
require 'puppet_x/util/orchestrator_error'
module PuppetX
  module Util
    class Orchestrator

      # Common routines used for interacting with the Orchestrator.
      class Connection

        # Service configuration Hash for the Orchestrator.
        # From services.conf, returned by
        # PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator'), for example.
        attr_accessor :service_config

        # RBAC token, can be read via PuppetX::Util::RBAC.load_token
        attr_accessor :rbac_token

        # Number of seconds to wait between Orchestrator job status checks.
        # (Mock to speed up specs...)
        def self.orch_status_wait_seconds
          1
        end

        def orch_status_wait_seconds
          self.class.orch_status_wait_seconds
        end

        # Convenience method for one off gets.
        #
        # @param service_config [Hash] as expected by a Connection instance.
        # @param rbac_token [String] as expected by a Connection instance.
        # @param path [String] /v1/some/thing that Orchestrator will recognize.
        # @param expected_code [Integer] the expected code from the request
        # (otherwise an error will be raised).
        # @return [[Hash,Array]] parsed response body.
        def self.get(service_config, rbac_token, path, expected_code: 200)
          trimmed_path = path[0] == '/' ? path[1..-1] : path
          url = "#{service_config[:url]}/#{trimmed_path}"
          uri = URI(url)
          impl = Connection.new(service_config, rbac_token)
          response = impl.orch_connection.get(uri, headers: impl.orch_request_headers)
          impl.check_orch_response(response, expected_code)
          JSON.parse(response.body)
        end

        def initialize(service_config, rbac_token)
          self.service_config = service_config
          self.rbac_token = rbac_token
        end

        def orch_connection
          if Puppet.runtime.instance_variable_get(:@runtime_services).keys.include? :http
            runtime_service = :http
          else
            runtime_service = 'http'
          end
          Puppet.runtime[runtime_service]
        end

        def orch_request_headers
          {'Content-Type' => 'application/json',
           'X-Authentication' => rbac_token,
           'User-Agent' => 'puppet-infrastructure'}
        end

        def check_orch_response(response, expected_http_status, scope: nil)
          begin
            parsed_body = JSON.parse(response.body || '{}')
          rescue JSON::ParserError
            parsed_body = {}
          end

          if response.code.to_i == 400 &&
             parsed_body['kind'] == 'puppetlabs.orchestrator/empty-target'
            raise PuppetX::Util::OrchestratorEmptyTargetError.new(scope)
          elsif response.code.to_i != expected_http_status
            raise PuppetX::Util::OrchestratorResponseError.new(response.code.to_i, response.body)
          elsif response.body.nil?
            raise PuppetX::Util::OrchestratorEmptyResponseError.new()
          end
        end
      end
    end
  end
end
