#! frozen_string_literal: true

require 'bolt/error'

# @summary This function is used to determine the platform name using os name, architecture, os version and fips.
#
# @param name The name of the os.
#
# @param arch The architecture of the  os.
#
# @param os_version The verision of the os.
#
# @param fips A boolean to determine if fips is enabled.
#
# @return [String] The platform name.
#
# @example
#   enterprise_platform::platform_name('debian', 'x86_64', '9')
#   Would return: 'debian-9-x86_64'
#
# @example
#   enterprise_platform::platform_name('windows', 'x64')
#   Would return: 'windows-x86_64'
#
# @example
#   enterprise_platform::platform_name('windows', 'x86')
#   Would return: 'windows-i386'
#
# @example
#   enterprise_platform::platform_name('el', 'x86_64', '7', true)
#   Would return: 'redhatfips-7-x86_64'

Puppet::Functions.create_function('enterprise_tasks::platform_name') do
  dispatch :platform_name do
    param 'String', :os
    param 'String', :arch
    param 'Optional[String]', :os_version
    param 'Boolean', :fips
    return_type 'Any'
  end

  def platform_name(os, arch, os_version = nil, fips = false)
    if fips
      case os
      when 'el'
        return "redhatfips-#{os_version}-#{arch}"
      when 'amazon'
        return "amazonfips-#{os_version}-#{arch}"
      when 'windows'
        return 'windowsfips-x86_64'
      end
    end
    return "#{os}-#{os_version}-#{arch}" unless os.match?(%r{^windows})
    case arch
    when 'x64'
      "#{os}-x86_64"
    when 'x86'
      "#{os}-i386"
    else
      raise Puppet::ParseErrorWithIssue.new("Architecture #{arch} is not supported for #{os}", 'puppetlabs.add-platform/invalid-input-error')
    end
  end
end
