# frozen_string_literal: true

require 'net/http'
require 'puppet/ssl'
require 'json'
require 'logging'

module PEBoltServer
  module Service
    class HttpClient
      attr_reader :client

      DEFAULT_HEADERS = { 'Content-Type': 'application/json' }.freeze

      def initialize(config)
        @logger = Bolt::Logger.logger(self)

        @logger.debug("Create SSL context")
        @ssl_context = Puppet::SSL::SSLProvider.new.create_context(
          cacerts: config[:ca_cert],
          crls: config[:crl],
          private_key: config[:private_key],
          client_cert: config[:cert]
        )

        # Taken from Puppet's Runtime. We do not want to look up
        # Puppet.runtime[:http] because it will share clients between processes.
        @client = Puppet::HTTP::Client.new
      end

      def get_with_cert(url_path, api_token = nil)
        headers = if api_token
                    DEFAULT_HEADERS.merge({ 'X-Authentication': api_token })
                  else
                    DEFAULT_HEADERS
                  end
        handle_request(url_path) do |parsed_uri|
          @client.get(
            parsed_uri,
            headers: headers,
            options: { ssl_context: @ssl_context }
          )
        end
      end

      private

      # Handle URL creation, JSON encoding/decoding, and errors for http requests
      def handle_request(url_path, body: nil)
        parsed_uri = URI(url_path)
        # Don't attempt to JSON.generate a nil body
        json_body = if body.nil?
                      nil
                    else
                      JSON.generate(body)
                    end

        http_response = if body.nil?
                          yield parsed_uri
                        else
                          yield parsed_uri, json_body
                        end

        # Throw and catch the response error. Throwing allows us to
        # capture if another part of the stack throws this error.
        unless http_response.success?
          raise Puppet::HTTP::ResponseError, Puppet::HTTP::ResponseError.new(http_response)
        end

        if http_response.body.nil? || http_response.body.empty?
          ["", http_response.code]
        else
          [JSON.parse(http_response.body), http_response.code]
        end
      rescue Puppet::HTTP::ResponseError => e
        msg = "API request to #{url_path} returned HTTP code #{e.response.code}"
        @logger.error(msg)
        # Use trace to print the message in case the request contained something sensitive
        @logger.trace("Failure message: #{e.full_message}")
        [msg, e.response.code]
      rescue StandardError => e
        msg = "Exception #{e.class.name} thrown while attempting API request to #{url_path}"
        @logger.error(msg)
        # Use trace to print the message in case the request contained something sensitive
        @logger.trace("Exception message: #{e.full_message}")
        [msg, 500]
      end
    end
  end
end
