require 'puppet'

module PeBackupTools
  module Utils
    # This module provides utilities to summarize the output of pbr commands
    module CommandSummary
      def self.summarize_create(info)
        # Now that we exit 1 immediately on errors, we should only arrive here if the status was successful
        scope_string = info[:scope].map { |s| "'#{s}'" }.join(', ').chomp
        message = _('Success! Backup of %{scope} is %{size}.') % { scope: scope_string, size: PeBackupTools::Utils.humanize_size(info[:size]) }
        message += "\n\n#{info[:filename]}\n\n"
        message += _('Move your backup to another location. Be sure to keep your backups in a secure, centralized location that is not on the primary.')
        Puppet.notice(message)
      end

      def self.summarize_restore(info)
        r10k_configured = info[:r10k_remote] && !info[:r10k_remote].empty?
        replica_present = info[:preexisting_replica] && !info[:preexisting_replica].empty?
        compilers_present = info[:compilers] && !info[:compilers].empty?

        # Now that we exit 1 immediately on errors, we should only arrive here if the status was successful
        message = _('Backup restored.')
        message += "\n  " + _('Time to restore: %{time}') % { time: PeBackupTools::Utils.humanize_time(info[:runtime][:total]) }
        message += "\n  " + _('Size: %{size}, Scope: %{scope}') % { size: PeBackupTools::Utils.humanize_size(info[:size]), scope: info[:scope].join(', ').chomp }
        message += "\n\nTo finish restoring your primary server from backup, run the following commands on this host:"
        if r10k_configured
          message += "\npuppet access login"
          message += "\npuppet code deploy --wait <environment> (Run for each environment you need to deploy, or use the --all flag)"
        end
        message += "\npuppet agent -t"
        message += "\npuppet agent -t"
        if compilers_present
          message += "\n\nThen, if you are restoring to the same operating system, on all compilers, run:"
          message += "\npuppet agent -t --server_list #{info[:restore_certname]}"
          unless replica_present
            message += "\npuppet agent -t"
          end
          message += "\nIf you are migrating to a new operating system, you will need to run 'puppet node purge' for your old compilers and provision new ones."
        end
        if replica_present
          message += "\n\nThen on this host:" if compilers_present
          message += "\npuppet infrastructure forget #{info[:preexisting_replica].first}"
          if compilers_present
            message += "\n\nThen on all compilers (only if restoring to the same operating system):"
            message += "\npuppet agent -t"
          end
        end
        Puppet.notice(message)
      end
    end
  end
end
