require 'puppet'
require 'puppet/util/pe_conf/recover'
require 'puppet/util/suidmanager'
require 'puppet_x/puppetlabs/meep/infra/lookup'
require 'puppet_x/puppetlabs/meep/puppet_context'
require 'erb'
require 'json'

class Puppet::Util::Pe_conf
  attr_accessor :pe_conf, :nodes_conf, :recovery_engine

  include PuppetX::Puppetlabs::Meep::PuppetContext
  include PuppetX::Puppetlabs::Meep::Infra::Lookup

  ENTERPRISE_DIR = '/etc/puppetlabs/enterprise'
  CONF_DIR = "#{ENTERPRISE_DIR}/conf.d"
  PE_CONF_PATH = "#{CONF_DIR}/pe.conf"
  NODE_CONF_PATH = "#{CONF_DIR}/nodes"

  def initialize
    @pe_conf = {}
    @nodes_conf = Hash.new{|h,k| h[k] = {}}
    @recovery_engine = Puppet::Util::Pe_conf::Recover.new
  end

  def default_environmentpath
    "#{Puppet[:codedir]}/environments"
  end

  # Creates a new pe.conf and associated nodes/certname.conf files
  # based on information from PuppetDB, the PE Classifier, and Hiera.
  #
  # << How it works
  #
  # << Limitations
  #
  # @param [Hash] opts Options to alter how information is gathered
  # @option opts [String] :pe_environmentpath The environmentpath to query against.
  # @option opts [String] :pe_environment The puppet environment to query against.
  def recover(opts = {})
    overrides = Hash.new{|h,k| h[k] = {}}
    if opts[:pe_environment].nil?
      begin
        default_env = recovery_engine.get_node_environment(Puppet[:certname])
        # If the node is in the agent-specified environment group, the node object
        # does not contain the environment_name value, so pull the environment
        # from puppet.conf.
        #
        # In some circumstances, we may be in an agent-specified environment group,
        # and the classifier terminus will delete the environment specified by the
        # the classifier and inject the environment we're making this request from,
        # which is "enterprise". This is an internal environment that doesn't exist
        # in the regular user environment location, and never what we want to use
        # here, so disregard this value and fall back to finding the environment
        # this alternate way.
        if default_env.nil? || default_env.empty? || default_env == 'enterprise'
          default_env = `/opt/puppetlabs/bin/puppet config print environment --section agent`.chomp
          default_env = default_env.empty? ? nil : default_env
        end
      rescue => e
        Puppet.err("Error getting node environment from classifier. Defaulting to 'production'. You may use the --pe-environment flag to bypass this check if 'production' is the wrong environment.")
        Puppet.err(e.full_message)
      end
    end
    default_env ||= 'production'
    env = opts[:pe_environment] || default_env
    environmentpath = opts[:pe_environmentpath] || default_environmentpath
    Puppet.notice(_("Using environment '#{env}' at #{environmentpath} to recover configuration"))

    puppet_environment_context(environmentpath, env, Puppet[:basemodulepath]) do
      _recover(env, opts[:target_dir], overrides)
    end
  end

  def _recover(env, target_dir, overrides)
    using_nc = recovery_engine.using_node_classifier?
    core_infra_nodes = recovery_engine.get_core_infra_nodes
    pe_conf = recovery_engine.pe_conf_from_puppetdb
    @pe_conf = recovery_engine.extract_required_pe_conf_params(pe_conf)
    node_terminus = recovery_engine.get_node_terminus
    external_nodes_setting = recovery_engine.get_external_nodes
    core_infra_nodes.each do |node_name|
      # Facts are needed for both classifier and hiera lookups
      facts = recovery_engine.facts_for_node(node_name)

      pe_module_resources = recovery_engine.resources_applied_to_node(node_name, 'Puppet_enterprise')
      pe_module_hiera_overrides = recovery_engine.find_hiera_overrides(node_name, pe_module_resources.keys, facts, env, node_terminus, external_nodes_setting)
      pe_module_hiera_overrides.each do |k,v|
        overrides[k][node_name] = v
      end

      pe_repo_resources = recovery_engine.resources_applied_to_node(node_name, 'Pe_repo')
      pe_repo_hiera_overrides = recovery_engine.find_hiera_overrides(node_name, pe_repo_resources.keys, facts, env, node_terminus, external_nodes_setting)
      pe_repo_hiera_overrides.each do |k,v|
        overrides[k][node_name] = v
      end

      classifier_data_backend = recovery_engine.find_hiera_overrides(node_name, ['puppet_enterprise_classifier_data_backend_present'], facts, env, node_terminus, external_nodes_setting)
      classifier_data_backend.each do |k,v|
        overrides[k][node_name] = v
      end

      # Grab any overrides that are specified in the classifier
      # (if the user is using it)
      if using_nc
        classifier_overrides = recovery_engine.classifier_overrides_for_node(node_name, facts, facts['trusted'])
        classifier_overrides.each do |k,v|
          overrides[k][node_name] = v
        end
      end
    end

    overrides.each do |param, nodes|
      # Create a global pe.conf override, if and only if, the value is the same
      # across all nodes **and** the override exists across all PE nodes.
      #
      # If you do not check that the override is on **all** nodes, then you
      # run the risk of adding a setting that is only actually provided on
      # one host becoming a global default. For an example, see PE-22999.
      #
      # Otherwise, the value should only be set for a specific node, so a
      # node-specific override should be set.
      if nodes.size == core_infra_nodes.size && nodes.values.uniq.length == 1
        @pe_conf[param] = nodes.values[0]
      else
        nodes.each do |node,value|
          @nodes_conf[node][param] = value
        end
      end
    end
  end
  private :_recover

  def render_conf_template(conf)
    template_path = File.join(File.expand_path(File.dirname(__FILE__)), 'pe_conf', 'templates', 'pe.conf.erb')
    template = ERB.new(File.read(template_path))
    template.result(binding)
  end

  def save(dir)
    user = 'pe-puppet'
    user_resource = resource_lookup('user', user)
    raise(RuntimeError, _("Unable to find the %{user} user. Unable to write config to %{dir}. %{user} resource: %{user_resource}") %  { user: user, dir: dir, user_resource: user_resource.inspect }) if user_resource[:ensure] != :present

    _save(dir, user_resource)
  end

  def _save(dir, user_resource)

    # In case /etc/puppetlabs/enterprise/conf.d doesn't exist, create it root:root,
    # and ensure it is initially readable by filesync watcher
    FileUtils.mkdir_p(dir, mode: 0755)
    # Ensure user has ownership of dir before we switch to user to modify it,
    # otherwise they may lack permissions to modify and fail.
    FileUtils.chown_R(user_resource[:uid], user_resource[:gid], dir)
    # Tighten permissions now that it is readable by filesync as pe-puppet...
    FileUtils.chmod(0700, dir)

    # Ensure /etc/puppetlabs/enterprise is owned by user if that is our parent dir.
    # (Installation of pe-modules which owns this directory, resets this to root,
    # which is an upgrade problem for recover_configuration...)
    if File.dirname(dir) == ENTERPRISE_DIR
      FileUtils.chown(user_resource[:uid], user_resource[:gid], ENTERPRISE_DIR)
    end

    Puppet::Util::SUIDManager.asuser(user_resource[:uid], user_resource[:gid]) do
      # Create a new user_data.conf with recovered values
      File.open("#{dir}/user_data.conf", 'w', 0600) do |file|
        file.write(render_conf_template(@pe_conf))
      end

      if @nodes_conf.size > 0
        nodes_dir = "#{dir}/nodes"
        FileUtils.mkdir_p(nodes_dir, mode: 0700)
        @nodes_conf.each do |node, overrides|
          node_conf_path = "#{dir}/nodes/#{node}.conf"
          # Write updated node overrides
          File.open(node_conf_path, 'w', 0600) do |file|
            file.write(render_conf_template(overrides))
          end
        end
      end
    end
  end

  # Used by the configure action when attempting recovery during upgrades.
  #
  # @param opts [Hash]
  # @option opts [String] :target_dir Target directory to save configuration
  #   to; defaults to CONF_DIR.
  # @option opts [String] :pe_environmentpath The environmentpath to recover from.
  # @option opts [String] :pe_environment The environment to recover from.
  # @return [Puppet::Util::Pe_conf] recovered configuration
  def self.recover_and_save_pe_configuration(opts = {})
    target_dir = opts[:target_dir] || CONF_DIR
    conf = self.new()
    conf.recover(opts.select { |k,v| [:pe_environmentpath, :pe_environment].include?(k) })
    conf.save(target_dir)
    return conf
  end
end
